local QBCore = exports['qb-core']:GetCoreObject()

print('^2[ICManager]^7 Server-side Lua script loaded')

-- Whitelist JSON file path
local resourceName = GetCurrentResourceName()
local whitelistPath = GetResourcePath(resourceName) .. '/lua/json/whitelisteds.json'
local banPath = GetResourcePath(resourceName) .. '/lua/json/banned.json'

-- Whitelist helper functions
local function LoadWhitelist()
    local file = io.open(whitelistPath, 'r')
    if not file then
        return {}
    end
    local content = file:read('*all')
    file:close()
    if content and content ~= '' then
        return json.decode(content) or {}
    end
    return {}
end

local function SaveWhitelist(whitelist)
    local file = io.open(whitelistPath, 'w')
    if not file then
        return false
    end
    file:write(json.encode(whitelist))
    file:close()
    return true
end

local function IsWhitelisted(discordId)
    local whitelist = LoadWhitelist()
    for i = 1, #whitelist do
        if whitelist[i].discordId == tostring(discordId) then
            return true, whitelist[i]
        end
    end
    return false, nil
end

local function AddToWhitelist(discordId, addedBy)
    local whitelist = LoadWhitelist()
    -- Check if already exists
    for i = 1, #whitelist do
        if whitelist[i].discordId == tostring(discordId) then
            return false, 'Discord ID already whitelisted'
        end
    end
    
    table.insert(whitelist, {
        discordId = tostring(discordId),
        addedBy = tostring(addedBy),
        addedAt = os.date('%Y-%m-%dT%H:%M:%S.000Z')
    })
    
    if SaveWhitelist(whitelist) then
        return true, nil
    else
        return false, 'Failed to save whitelist'
    end
end

local function RemoveFromWhitelist(discordId)
    local whitelist = LoadWhitelist()
    local found = false
    
    for i = #whitelist, 1, -1 do
        if whitelist[i].discordId == tostring(discordId) then
            table.remove(whitelist, i)
            found = true
        end
    end
    
    if found then
        if SaveWhitelist(whitelist) then
            return true, nil
        else
            return false, 'Failed to save whitelist'
        end
    else
        return false, 'Discord ID not found in whitelist'
    end
end

-- Ban system helper functions
local function LoadBans()
    local file = io.open(banPath, 'r')
    if not file then
        return {}
    end
    local content = file:read('*all')
    file:close()
    if content and content ~= '' then
        return json.decode(content) or {}
    end
    return {}
end

local function SaveBans(bans)
    local file = io.open(banPath, 'w')
    if not file then
        return false
    end
    file:write(json.encode(bans))
    file:close()
    return true
end

local function GetBanDurationInSeconds(banDuration)
    if banDuration == '1 Day' then
        return 24 * 60 * 60
    elseif banDuration == '3 Day' then
        return 3 * 24 * 60 * 60
    elseif banDuration == '7 Day' then
        return 7 * 24 * 60 * 60
    elseif banDuration == '1 Month' then
        return 30 * 24 * 60 * 60
    elseif banDuration == 'Permanent' then
        return nil -- nil means permanent
    else
        return nil
    end
end

local function IsBanned(discordId)
    local bans = LoadBans()
    for i = 1, #bans do
        local ban = bans[i]
        if ban.discordId == tostring(discordId) and not ban.isUnbanned then
            -- Check if ban is permanent
            if ban.banDuration == 'Permanent' or not ban.expiresAt then
                return true, ban
            end
            
            -- Check if ban has expired
            local currentTime = os.time()
            local expiresAt = tonumber(ban.expiresAt)
            if expiresAt and currentTime < expiresAt then
                return true, ban
            else
                -- Ban has expired, mark as unbanned automatically
                ban.isUnbanned = true
                ban.unbannedAt = os.date('%Y-%m-%dT%H:%M:%S.000Z')
                ban.unbanReason = 'Ban süresi doldu'
                ban.unbannedBy = 'System'
                SaveBans(bans)
                return false, nil
            end
        end
    end
    return false, nil
end

local function BanPlayer(discordId, banReason, banDuration, bannedBy)
    local bans = LoadBans()
    
    -- Check if already banned and not unbanned
    for i = 1, #bans do
        if bans[i].discordId == tostring(discordId) and not bans[i].isUnbanned then
            return false, 'Discord ID already banned'
        end
    end
    
    local bannedAt = os.time()
    local expiresAt = nil
    local durationSeconds = GetBanDurationInSeconds(banDuration)
    
    if durationSeconds then
        expiresAt = bannedAt + durationSeconds
    end
    
    table.insert(bans, {
        discordId = tostring(discordId),
        banReason = tostring(banReason),
        banDuration = tostring(banDuration),
        bannedAt = os.date('%Y-%m-%dT%H:%M:%S.000Z', bannedAt),
        bannedBy = tostring(bannedBy),
        expiresAt = expiresAt,
        isUnbanned = false,
        unbannedAt = nil,
        unbanReason = nil,
        unbannedBy = nil
    })
    
    if SaveBans(bans) then
        -- Check if player is online and kick them
        local players = GetPlayers()
        for _, playerId in ipairs(players) do
            local playerSrc = tonumber(playerId)
            if playerSrc then
                local identifiers = GetPlayerIdentifiers(playerSrc)
                local playerDiscordId = nil
                
                -- Find discord identifier
                for j = 1, #identifiers do
                    if string.find(identifiers[j], 'discord:') then
                        playerDiscordId = string.gsub(identifiers[j], 'discord:', '')
                        break
                    end
                end
                
                -- If this player matches the banned discord ID, kick them
                if playerDiscordId and tostring(playerDiscordId) == tostring(discordId) then
                    local banMessage = 'Oyundan yasaklandınız.\nSebep: ' .. banReason
                    
                    if banDuration == 'Permanent' or not expiresAt then
                        banMessage = banMessage .. '\nSüre: Kalıcı'
                    else
                        local remainingSeconds = expiresAt - os.time()
                        local remainingDays = math.floor(remainingSeconds / (24 * 60 * 60))
                        local remainingHours = math.floor((remainingSeconds % (24 * 60 * 60)) / (60 * 60))
                        local remainingMinutes = math.floor((remainingSeconds % (60 * 60)) / 60)
                        banMessage = banMessage .. string.format('\nKalan Süre: %d gün, %d saat, %d dakika', remainingDays, remainingHours, remainingMinutes)
                    end
                    
                    banMessage = banMessage .. '\nYasaklayan: ' .. bannedBy
                    
                    DropPlayer(playerSrc, banMessage)
                    print(string.format('^2[ICManager]^7 Kicked player %d (Discord: %s) due to ban', playerSrc, discordId))
                end
            end
        end
        
        return true, nil
    else
        return false, 'Failed to save ban'
    end
end

local function UnbanPlayer(discordId, unbanReason, unbannedBy)
    local bans = LoadBans()
    local found = false
    
    for i = 1, #bans do
        if bans[i].discordId == tostring(discordId) and not bans[i].isUnbanned then
            bans[i].isUnbanned = true
            bans[i].unbannedAt = os.date('%Y-%m-%dT%H:%M:%S.000Z')
            bans[i].unbanReason = tostring(unbanReason)
            bans[i].unbannedBy = tostring(unbannedBy)
            found = true
        end
    end
    
    if found then
        if SaveBans(bans) then
            return true, nil
        else
            return false, 'Failed to save ban'
        end
    else
        return false, 'Discord ID not found in bans or already unbanned'
    end
end

local function GetBanInfo(discordId)
    local bans = LoadBans()
    for i = 1, #bans do
        if bans[i].discordId == tostring(discordId) then
            local ban = bans[i]
            local banInfo = {
                discordId = ban.discordId,
                banReason = ban.banReason,
                banDuration = ban.banDuration,
                bannedAt = ban.bannedAt,
                bannedBy = ban.bannedBy,
                expiresAt = ban.expiresAt,
                isUnbanned = ban.isUnbanned,
                unbannedAt = ban.unbannedAt,
                unbanReason = ban.unbanReason,
                unbannedBy = ban.unbannedBy
            }
            
            -- Calculate remaining time if not unbanned and not permanent
            if not ban.isUnbanned and ban.expiresAt then
                local currentTime = os.time()
                local expiresAt = tonumber(ban.expiresAt)
                if expiresAt and currentTime < expiresAt then
                    local remainingSeconds = expiresAt - currentTime
                    local remainingDays = math.floor(remainingSeconds / (24 * 60 * 60))
                    local remainingHours = math.floor((remainingSeconds % (24 * 60 * 60)) / (60 * 60))
                    local remainingMinutes = math.floor((remainingSeconds % (60 * 60)) / 60)
                    banInfo.remainingTime = string.format('%d gün, %d saat, %d dakika', remainingDays, remainingHours, remainingMinutes)
                    banInfo.isExpired = false
                else
                    banInfo.isExpired = true
                end
            elseif not ban.isUnbanned and not ban.expiresAt then
                banInfo.isExpired = false
                banInfo.remainingTime = 'Kalıcı'
            end
            
            return true, banInfo
        end
    end
    return false, nil
end

-- Player Connecting Check
AddEventHandler('playerConnecting', function(name, setKickReason, deferrals)
    local src = source
    local identifiers = GetPlayerIdentifiers(src)
    local discordId = nil
    
    -- Find discord identifier
    for i = 1, #identifiers do
        if string.find(identifiers[i], 'discord:') then
            discordId = string.gsub(identifiers[i], 'discord:', '')
            break
        end
    end
    
    -- Check if discord is required
    if not discordId or discordId == '' then
        deferrals.done('Discord\'un açık olması zorunludur')
        return
    end
    
    -- Check whitelist
    local isWhitelisted, whitelistData = IsWhitelisted(discordId)
    if not isWhitelisted then
        deferrals.done('Bu sunucuda whitelistiniz yok')
        return
    end
    
    -- Check ban
    local isBanned, banData = IsBanned(discordId)
    if isBanned and banData then
        local banMessage = 'Sunucudan yasaklandınız.\nSebep: ' .. banData.banReason
        
        if banData.banDuration == 'Permanent' or not banData.expiresAt then
            banMessage = banMessage .. '\nSüre: Kalıcı'
        else
            local currentTime = os.time()
            local expiresAt = tonumber(banData.expiresAt)
            if expiresAt then
                local remainingSeconds = expiresAt - currentTime
                local remainingDays = math.floor(remainingSeconds / (24 * 60 * 60))
                local remainingHours = math.floor((remainingSeconds % (24 * 60 * 60)) / (60 * 60))
                local remainingMinutes = math.floor((remainingSeconds % (60 * 60)) / 60)
                banMessage = banMessage .. string.format('\nKalan Süre: %d gün, %d saat, %d dakika', remainingDays, remainingHours, remainingMinutes)
            end
        end
        
        -- Format tarih
        local formattedDate = banData.bannedAt
        if banData.bannedAt then
            -- Parse ISO date string (2025-11-06T16:45:05.000Z)
            local year, month, day, hour, minute = string.match(banData.bannedAt, '(%d+)-(%d+)-(%d+)T(%d+):(%d+):')
            if year and month and day and hour and minute then
                local monthNames = {'Ocak', 'Şubat', 'Mart', 'Nisan', 'Mayıs', 'Haziran', 'Temmuz', 'Ağustos', 'Eylül', 'Ekim', 'Kasım', 'Aralık'}
                local monthName = monthNames[tonumber(month)]
                formattedDate = string.format('%s %s %s, %s:%s', day, monthName, year, hour, minute)
            end
        end
        
        banMessage = banMessage .. '\nYasaklayan: ' .. banData.bannedBy
        banMessage = banMessage .. '\nTarih: ' .. formattedDate
        
        deferrals.done(banMessage)
        return
    end
    
    -- Allow connection
    deferrals.done()
end)

local function GetPlayersFromDB()
    local players = MySQL.query.await('SELECT * FROM players', {})
    return players
end

-- Export function for Node.js to call
exports('getPlayersFromDB', function(callback)
    if not callback then
        print('^1[ICManager]^7 Error: No callback provided to getPlayersFromDB')
        return
    end
    
    CreateThread(function()
        local success, result = pcall(GetPlayersFromDB)
        
        if success and result then
            callback(nil, result)
            print('^2[ICManager]^7 Successfully fetched ' .. #result .. ' players from database')
        else
            local errorMsg = result or 'Unknown error occurred'
            print('^1[ICManager]^7 Error fetching players: ' .. tostring(errorMsg))
            callback(tostring(errorMsg), nil)
        end
    end)
end)

-- Helper function to get player by ID
local function GetPlayer(src)
    local Player = QBCore.Functions.GetPlayer(src)
    if not Player then
        return nil, 'Player not found'
    end
    return Player, nil
end

-- Set Player Money
exports('setPlayerMoney', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local src = tonumber(params.playerId)
        local moneyType = params.moneyType
        local amount = tonumber(params.amount)
        
        if not src or not moneyType or not amount then
            callback('Invalid parameters', nil)
            return
        end
        
        local Player, error = GetPlayer(src)
        if not Player then
            callback(error, nil)
            return
        end
        
        local success, err = pcall(function()
            Player.Functions.SetMoney(moneyType, amount)
        end)
        
        if success then
            callback(nil, { success = true, message = 'Money set successfully' })
            print(string.format('^2[ICManager]^7 Set %s money to %d for player %d', moneyType, amount, src))
        else
            callback(tostring(err), nil)
        end
    end)
end)

-- Add Player Money
exports('addPlayerMoney', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local src = tonumber(params.playerId)
        local moneyType = params.moneyType
        local amount = tonumber(params.amount)
        
        if not src or not moneyType or not amount then
            callback('Invalid parameters', nil)
            return
        end
        
        local Player, error = GetPlayer(src)
        if not Player then
            callback(error, nil)
            return
        end
        
        local success, err = pcall(function()
            Player.Functions.AddMoney(moneyType, amount)
        end)
        
        if success then
            callback(nil, { success = true, message = 'Money added successfully' })
            print(string.format('^2[ICManager]^7 Added %d %s money to player %d', amount, moneyType, src))
        else
            callback(tostring(err), nil)
        end
    end)
end)

-- Remove Player Money
exports('removePlayerMoney', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local src = tonumber(params.playerId)
        local moneyType = params.moneyType
        local amount = tonumber(params.amount)
        
        if not src or not moneyType or not amount then
            callback('Invalid parameters', nil)
            return
        end
        
        local Player, error = GetPlayer(src)
        if not Player then
            callback(error, nil)
            return
        end
        
        local success, err = pcall(function()
            Player.Functions.RemoveMoney(moneyType, amount)
        end)
        
        if success then
            callback(nil, { success = true, message = 'Money removed successfully' })
            print(string.format('^2[ICManager]^7 Removed %d %s money from player %d', amount, moneyType, src))
        else
            callback(tostring(err), nil)
        end
    end)
end)

-- Add Player Item
exports('addPlayerItem', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local src = tonumber(params.playerId)
        local item = params.item
        local itemCount = tonumber(params.itemCount) or 1
        
        if not src or not item then
            callback('Invalid parameters', nil)
            return
        end
        
        local Player, error = GetPlayer(src)
        if not Player then
            callback(error, nil)
            return
        end
        
        local success, err = pcall(function()
            Player.Functions.AddItem(item, itemCount)
        end)
        
        if success then
            callback(nil, { success = true, message = 'Item added successfully' })
            print(string.format('^2[ICManager]^7 Added %d x %s to player %d', itemCount, item, src))
        else
            callback(tostring(err), nil)
        end
    end)
end)

-- Remove Player Item
exports('removePlayerItem', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local src = tonumber(params.playerId)
        local item = params.item
        local itemCount = tonumber(params.itemCount) or 1
        
        if not src or not item then
            callback('Invalid parameters', nil)
            return
        end
        
        local Player, error = GetPlayer(src)
        if not Player then
            callback(error, nil)
            return
        end
        
        local success, err = pcall(function()
            Player.Functions.RemoveItem(item, itemCount)
        end)
        
        if success then
            callback(nil, { success = true, message = 'Item removed successfully' })
            print(string.format('^2[ICManager]^7 Removed %d x %s from player %d', itemCount, item, src))
        else
            callback(tostring(err), nil)
        end
    end)
end)

-- Revive Player
exports('revivePlayer', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local src = tonumber(params.playerId)
        
        if not src then
            callback('Invalid playerId', nil)
            return
        end
        
        local Player, error = GetPlayer(src)
        if not Player then
            callback(error, nil)
            return
        end
        
        local success, err = pcall(function()
            TriggerClientEvent('hospital:client:Revive', src)
        end)
        
        if success then
            callback(nil, { success = true, message = 'Player revived successfully' })
            print(string.format('^2[ICManager]^7 Revived player %d', src))
        else
            callback(tostring(err), nil)
        end
    end)
end)

-- Kill Player
exports('killPlayer', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local src = tonumber(params.playerId)
        
        if not src then
            callback('Invalid playerId', nil)
            return
        end
        
        local Player, error = GetPlayer(src)
        if not Player then
            callback(error, nil)
            return
        end
        
        local success, err = pcall(function()
            TriggerClientEvent('hospital:client:KillPlayer', src)
        end)
        
        if success then
            callback(nil, { success = true, message = 'Player killed successfully' })
            print(string.format('^2[ICManager]^7 Killed player %d', src))
        else
            callback(tostring(err), nil)
        end
    end)
end)

-- Generate unique plate
local function generatePlate()
    local plate = QBCore.Shared.RandomInt(1) .. QBCore.Shared.RandomStr(2) .. QBCore.Shared.RandomInt(3) .. QBCore.Shared.RandomStr(2)
    local result = MySQL.scalar.await('SELECT plate FROM player_vehicles WHERE plate = ?', { plate })
    if result then
        return generatePlate()
    else
        return plate:upper()
    end
end

-- Set Player Job
exports('setPlayerJob', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local src = tonumber(params.playerId)
        local jobName = params.jobName
        local jobGrade = tonumber(params.jobGrade)
        
        if not src or not jobName or not jobGrade then
            callback('Invalid parameters', nil)
            return
        end
        
        local Player, error = GetPlayer(src)
        if not Player then
            callback(error, nil)
            return
        end
        
        local success, err = pcall(function()
            Player.Functions.SetJob(jobName, jobGrade)
        end)
        
        if success then
            callback(nil, { success = true, message = 'Job set successfully', job = jobName, grade = jobGrade })
            print(string.format('^2[ICManager]^7 Set job %s (grade %d) for player %d', jobName, jobGrade, src))
        else
            callback(tostring(err), nil)
        end
    end)
end)

-- Give Vehicle to Player
exports('givePlayerVehicle', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local src = tonumber(params.playerId)
        local carName = params.carName
        local customPlate = params.plate or nil
        local fullmod = params.fullmod or false
        local renk1 = params.renk1 or params.color1 or "0 0 0"
        local renk2 = params.renk2 or params.color2 or "0 0 0"
        
        if not src or not carName then
            callback('Invalid parameters', nil)
            return
        end
        
        local Player, error = GetPlayer(src)
        if not Player then
            callback(error, nil)
            return
        end
        
        local success, err = pcall(function()
            -- Generate or use custom plate
            local plate = (customPlate and customPlate ~= "") and string.upper(customPlate) or string.upper(generatePlate())
            
            -- Check if plate already exists
            local existingPlate = MySQL.scalar.await('SELECT plate FROM player_vehicles WHERE plate = ?', { plate })
            if existingPlate then
                if customPlate then
                    return nil, 'This plate already exists!'
                else
                    -- Regenerate if auto-generated plate exists
                    plate = string.upper(generatePlate())
                end
            end
            
            local cid = Player.PlayerData.citizenid
            local vehicleHash = GetHashKey(carName)
            
            MySQL.insert('INSERT INTO player_vehicles (license, citizenid, vehicle, hash, mods, plate, state) VALUES (?, ?, ?, ?, ?, ?, ?)', {
                Player.PlayerData.license,
                cid,
                carName,
                vehicleHash,
                '{}',
                plate,
                0
            })
            
            TriggerClientEvent('icmanager:client:getvehicle', src, carName, renk1, renk2, fullmod, plate)
            
            return { success = true, plate = plate, vehicle = carName }
        end)
        
        if success and err then
            callback(nil, err)
            print(string.format('^2[ICManager]^7 Gave vehicle %s (plate: %s) to player %d', carName, err.plate, src))
        else
            callback(tostring(err) or 'Failed to give vehicle', nil)
        end
    end)
end)

exports('getPlayerInventory', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local src = tonumber(params.playerId)
        
        if not src then
            callback('Invalid playerId', nil)
            return
        end
        
        local Player, error = GetPlayer(src)
        if not Player then
            callback(error, nil)
            return
        end
        
        local success, err = pcall(function()
            local inventory = {}
            local items = Player.PlayerData.items
            
            -- QBCore items can be a table or array
            if type(items) == 'table' then
                for slot, item in pairs(items) do
                    if item and item.name then
                        local itemData = QBCore.Shared.Items[item.name]
                        
                        if itemData then
                            table.insert(inventory, {
                                name = item.name,
                                label = itemData.label or item.name,
                                count = item.amount or item.count or 0,
                                item = item.name,
                                description = itemData.description or ''
                            })
                        end
                    end
                end
            end
            
            return inventory
        end)
        
        if success and err then
            callback(nil, err)
            print(string.format('^2[ICManager]^7 Retrieved inventory for player %d (%d items)', src, #err))
        else
            callback(tostring(err), nil)
        end
    end)
end)

-- Add to Whitelist
exports('addToWhitelist', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local discordId = params.discordId
        local addedBy = params.addedBy
        
        if not discordId or not addedBy then
            callback('Invalid parameters', nil)
            return
        end
        
        local success, error = AddToWhitelist(discordId, addedBy)
        
        if success then
            callback(nil, { success = true, message = 'Added to whitelist', discordId = discordId })
            print(string.format('^2[ICManager]^7 Added %s to whitelist by %s', discordId, addedBy))
        else
            callback(error or 'Failed to add to whitelist', nil)
        end
    end)
end)

-- Remove from Whitelist
exports('removeFromWhitelist', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local discordId = params.discordId
        local removedBy = params.removedBy
        
        if not discordId or not removedBy then
            callback('Invalid parameters', nil)
            return
        end
        
        local success, error = RemoveFromWhitelist(discordId)
        
        if success then
            callback(nil, { success = true, message = 'Removed from whitelist', discordId = discordId })
            print(string.format('^2[ICManager]^7 Removed %s from whitelist by %s', discordId, removedBy))
        else
            callback(error or 'Failed to remove from whitelist', nil)
        end
    end)
end)

-- Get Whitelist Info
exports('getWhitelistInfo', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local discordId = params.discordId
        
        if not discordId then
            callback('Invalid parameters', nil)
            return
        end
        
        local isWhitelisted, whitelistData = IsWhitelisted(discordId)
        
        if isWhitelisted and whitelistData then
            callback(nil, {
                discordId = whitelistData.discordId,
                addedBy = whitelistData.addedBy,
                addedAt = whitelistData.addedAt
            })
        else
            callback('Discord ID not found in whitelist', nil)
        end
    end)
end)

-- Ban Player
exports('banPlayer', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local discordId = params.discordId
        local banReason = params.banReason
        local banDuration = params.banDuration
        local bannedBy = params.bannedBy
        
        if not discordId or not banReason or not banDuration or not bannedBy then
            callback('Invalid parameters', nil)
            return
        end
        
        -- Validate ban duration
        local validDurations = { '1 Day', '3 Day', '7 Day', '1 Month', 'Permanent' }
        local isValidDuration = false
        for i = 1, #validDurations do
            if banDuration == validDurations[i] then
                isValidDuration = true
                break
            end
        end
        
        if not isValidDuration then
            callback('Invalid ban duration. Must be: 1 Day, 3 Day, 7 Day, 1 Month, or Permanent', nil)
            return
        end
        
        local success, error = BanPlayer(discordId, banReason, banDuration, bannedBy)
        
        if success then
            callback(nil, { success = true, message = 'Player banned successfully', discordId = discordId })
            print(string.format('^2[ICManager]^7 Banned %s by %s for %s', discordId, bannedBy, banDuration))
        else
            callback(error or 'Failed to ban player', nil)
        end
    end)
end)

-- Unban Player
exports('unbanPlayer', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local discordId = params.discordId
        local unbanReason = params.unbanReason
        local unbannedBy = params.unbannedBy
        
        if not discordId or not unbanReason or not unbannedBy then
            callback('Invalid parameters', nil)
            return
        end
        
        local success, error = UnbanPlayer(discordId, unbanReason, unbannedBy)
        
        if success then
            callback(nil, { success = true, message = 'Player unbanned successfully', discordId = discordId })
            print(string.format('^2[ICManager]^7 Unbanned %s by %s', discordId, unbannedBy))
        else
            callback(error or 'Failed to unban player', nil)
        end
    end)
end)

-- Get Ban Info
exports('getBanInfo', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local discordId = params.discordId
        
        if not discordId then
            callback('Invalid parameters', nil)
            return
        end
        
        local found, banInfo = GetBanInfo(discordId)
        
        if found and banInfo then
            callback(nil, banInfo)
        else
            callback('Discord ID not found in bans', nil)
        end
    end)
end)


-- Get Player Info
exports('getPlayerInfo', function(params, callback)
    if not callback then return end
    
    CreateThread(function()
        local src = tonumber(params.playerId)
        
        if not src then
            callback('Invalid parameters', nil)
            return
        end
        
        local Player, error = GetPlayer(src)
        if not Player then
            callback(error, nil)
            return
        end
        
        local success, err = pcall(function()
            local playerData = Player.PlayerData
            
            -- Get player name
            local firstName = playerData.charinfo.firstname or 'N/A'
            local lastName = playerData.charinfo.lastname or 'N/A'
            
            -- Get job info
            local jobName = playerData.job.name or 'unemployed'
            local jobLabel = playerData.job.label or 'Unemployed'
            local jobGrade = playerData.job.grade.level or 0
            local jobGradeLabel = playerData.job.grade.name or 'Grade ' .. jobGrade
            
            -- Get money
            local cash = playerData.money.cash or 0
            local bank = playerData.money.bank or 0
            
            -- Get all identifiers
            local identifiers = GetPlayerIdentifiers(src)
            local identifierList = {}
            for i = 1, #identifiers do
                table.insert(identifierList, identifiers[i])
            end
            
            return {
                firstName = firstName,
                lastName = lastName,
                fullName = firstName .. ' ' .. lastName,
                job = {
                    name = jobName,
                    label = jobLabel,
                    grade = jobGrade,
                    gradeLabel = jobGradeLabel
                },
                money = {
                    cash = cash,
                    bank = bank
                },
                identifiers = identifierList
            }
        end)
        
        if success and err then
            callback(nil, err)
            print(string.format('^2[ICManager]^7 Retrieved player info for player %d', src))
        else
            callback(tostring(err) or 'Failed to get player info', nil)
        end
    end)
end)
