const fs = require('fs');
const path = require('path');
const axios = require('axios');
const FormData = require('form-data');

class TranscriptSistemi {
    constructor() {
        this.uploadApiUrl = 'http://cdn.hexatheme.com/upload_api.php';
    }

    async generateTranscript(channel, ticketOwner) {
        try {
            console.log(`Transcript oluşturuluyor: ${channel.name}`);
            
            const messages = await this.fetchAllMessages(channel);
            console.log(`${messages.length} mesaj bulundu`);

            const processedMessages = await this.processMessagesWithAttachments(messages);
            const html = this.generateHTML(processedMessages, channel, ticketOwner);
            const transcriptResult = await this.uploadTranscript(html, channel.name);
            
            return {
                url: transcriptResult.url,
                viewUrl: transcriptResult.viewUrl,
                messageCount: messages.length,
                attachmentCount: this.countAttachments(messages)
            };
        } catch (error) {
            console.error('Transcript oluşturma hatası:', error);
            throw error;
        }
    }

    async fetchAllMessages(channel) {
        const messages = [];
        let lastMessageId = null;

        while (true) {
            const options = { limit: 100 };
            if (lastMessageId) {
                options.before = lastMessageId;
            }

            const batch = await channel.messages.fetch(options);
            if (batch.size === 0) break;

            messages.push(...batch.values());
            lastMessageId = batch.last().id;
        }

        return messages.reverse();
    }

    async processMessagesWithAttachments(messages) {
        const processedMessages = [];

        for (const message of messages) {
            const processedMessage = {
                id: message.id,
                author: {
                    username: message.author.username,
                    discriminator: message.author.discriminator,
                    tag: message.author.tag,
                    avatar: message.author.displayAvatarURL({ dynamic: true, size: 128 }),
                    bot: message.author.bot
                },
                content: message.content,
                timestamp: message.createdTimestamp,
                editedTimestamp: message.editedTimestamp,
                attachments: [],
                embeds: message.embeds,
                reactions: message.reactions.cache.map(reaction => ({
                    emoji: reaction.emoji.name || reaction.emoji.toString(),
                    count: reaction.count
                }))
            };

            if (message.attachments.size > 0) {
                for (const attachment of message.attachments.values()) {
                    try {
                        const uploadedFile = await this.uploadAttachment(attachment);
                        processedMessage.attachments.push({
                            name: attachment.name,
                            size: attachment.size,
                            url: uploadedFile.url,
                            originalUrl: attachment.url,
                            contentType: attachment.contentType
                        });
                    } catch (error) {
                        console.error(`Dosya yükleme hatası: ${attachment.name}`, error);
                        processedMessage.attachments.push({
                            name: attachment.name,
                            size: attachment.size,
                            url: attachment.url,
                            originalUrl: attachment.url,
                            contentType: attachment.contentType,
                            uploadError: true
                        });
                    }
                }
            }

            processedMessages.push(processedMessage);
        }

        return processedMessages;
    }

    async uploadAttachment(attachment) {
        try {
            const response = await axios.get(attachment.url, { responseType: 'stream' });
            
            const formData = new FormData();
            formData.append('file', response.data, {
                filename: attachment.name,
                contentType: attachment.contentType
            });

            const uploadResponse = await axios.post(this.uploadApiUrl, formData, {
                headers: {
                    ...formData.getHeaders()
                }
            });

            if (uploadResponse.data.success) {
                return uploadResponse.data.data;
            } else {
                throw new Error(uploadResponse.data.message);
            }
        } catch (error) {
            console.error('Dosya yükleme hatası:', error);
            throw error;
        }
    }

    generateHTML(messages, channel, ticketOwner) {
        const html = `
<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Ticket Transcript - ${channel.name}</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #36393f;
            color: #dcddde;
            line-height: 1.4;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }

        .header {
            background: linear-gradient(135deg, #7289da, #5865f2);
            padding: 30px;
            border-radius: 10px;
            margin-bottom: 30px;
            text-align: center;
            box-shadow: 0 4px 15px rgba(0, 0, 0, 0.3);
        }

        .header h1 {
            font-size: 2.5em;
            margin-bottom: 10px;
            color: white;
        }

        .header .info {
            font-size: 1.1em;
            opacity: 0.9;
            color: white;
        }

        .messages {
            background-color: #40444b;
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 2px 10px rgba(0, 0, 0, 0.2);
        }

        .message {
            display: flex;
            margin-bottom: 20px;
            padding: 15px;
            border-radius: 8px;
            transition: background-color 0.2s;
        }

        .message:hover {
            background-color: #32353b;
        }

        .message.bot {
            background-color: #2f3136;
        }

        .avatar {
            width: 50px;
            height: 50px;
            border-radius: 50%;
            margin-right: 15px;
            flex-shrink: 0;
        }

        .message-content {
            flex: 1;
        }

        .message-header {
            display: flex;
            align-items: center;
            margin-bottom: 8px;
        }

        .username {
            font-weight: 600;
            color: #ffffff;
            margin-right: 10px;
        }

        .username.bot {
            color: #5865f2;
        }

        .timestamp {
            font-size: 0.85em;
            color: #72767d;
        }

        .edited {
            font-size: 0.8em;
            color: #72767d;
            font-style: italic;
            margin-left: 5px;
        }

        .message-text {
            color: #dcddde;
            word-wrap: break-word;
            margin-bottom: 10px;
        }

        .attachments {
            margin-top: 10px;
        }

        .attachment {
            display: inline-block;
            background-color: #2f3136;
            border: 1px solid #40444b;
            border-radius: 6px;
            padding: 10px;
            margin: 5px 5px 5px 0;
            text-decoration: none;
            color: #00aff4;
            transition: background-color 0.2s;
        }

        .attachment:hover {
            background-color: #36393f;
        }

        .attachment.image {
            display: block;
            max-width: 400px;
            margin: 10px 0;
        }

        .attachment.image img {
            max-width: 100%;
            border-radius: 6px;
        }

        .reactions {
            margin-top: 10px;
        }

        .reaction {
            display: inline-block;
            background-color: #2f3136;
            border: 1px solid #40444b;
            border-radius: 12px;
            padding: 4px 8px;
            margin: 2px;
            font-size: 0.9em;
        }

        .embed {
            border-left: 4px solid #5865f2;
            background-color: #2f3136;
            padding: 15px;
            margin: 10px 0;
            border-radius: 0 6px 6px 0;
        }

        .footer {
            text-align: center;
            margin-top: 30px;
            padding: 20px;
            color: #72767d;
            font-size: 0.9em;
        }

        @media (max-width: 768px) {
            .container {
                padding: 10px;
            }
            
            .header h1 {
                font-size: 2em;
            }
            
            .message {
                padding: 10px;
            }
            
            .avatar {
                width: 40px;
                height: 40px;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>🎫 Ticket Transcript</h1>
            <div class="info">
                <strong>Kanal:</strong> #${channel.name}<br>
                <strong>Ticket Sahibi:</strong> ${ticketOwner ? ticketOwner.tag : 'Bilinmeyen Kullanıcı'}<br>
                <strong>Oluşturulma Tarihi:</strong> ${new Date().toLocaleString('tr-TR')}<br>
                <strong>Toplam Mesaj:</strong> ${messages.length}
            </div>
        </div>

        <div class="messages">
            ${messages.map(message => this.generateMessageHTML(message)).join('')}
        </div>

        <div class="footer">
            <p>Bu transcript RoveBot's Service's Ticket Sistemi tarafından otomatik olarak oluşturulmuştur.</p>
            <p>Oluşturulma Zamanı: ${new Date().toLocaleString('tr-TR')}</p>
        </div>
    </div>
</body>
</html>`;

        return html;
    }

    generateMessageHTML(message) {
        const date = new Date(message.timestamp);
        const timeString = date.toLocaleString('tr-TR');
        const botClass = message.author.bot ? ' bot' : '';
        const usernameClass = message.author.bot ? ' bot' : '';

        let attachmentsHTML = '';
        if (message.attachments.length > 0) {
            attachmentsHTML = `
                <div class="attachments">
                    ${message.attachments.map(att => {
                        if (att.contentType && att.contentType.startsWith('image/')) {
                            return `
                                <a href="${att.url}" target="_blank" class="attachment image">
                                    <img src="${att.url}" alt="${att.name}" />
                                    <div>${att.name} (${this.formatFileSize(att.size)})</div>
                                </a>
                            `;
                        } else {
                            return `
                                <a href="${att.url}" target="_blank" class="attachment">
                                    📎 ${att.name} (${this.formatFileSize(att.size)})
                                    ${att.uploadError ? ' ⚠️ Yükleme hatası' : ''}
                                </a>
                            `;
                        }
                    }).join('')}
                </div>
            `;
        }

        let reactionsHTML = '';
        if (message.reactions.length > 0) {
            reactionsHTML = `
                <div class="reactions">
                    ${message.reactions.map(reaction => 
                        `<span class="reaction">${reaction.emoji} ${reaction.count}</span>`
                    ).join('')}
                </div>
            `;
        }

        let embedsHTML = '';
        if (message.embeds.length > 0) {
            embedsHTML = message.embeds.map(embed => `
                <div class="embed">
                    ${embed.title ? `<strong>${embed.title}</strong><br>` : ''}
                    ${embed.description ? `${embed.description}<br>` : ''}
                </div>
            `).join('');
        }

        return `
            <div class="message${botClass}">
                <img src="${message.author.avatar}" alt="${message.author.username}" class="avatar">
                <div class="message-content">
                    <div class="message-header">
                        <span class="username${usernameClass}">${message.author.username}</span>
                        <span class="timestamp">${timeString}</span>
                        ${message.editedTimestamp ? '<span class="edited">(düzenlendi)</span>' : ''}
                    </div>
                    ${message.content ? `<div class="message-text">${this.escapeHtml(message.content)}</div>` : ''}
                    ${embedsHTML}
                    ${attachmentsHTML}
                    ${reactionsHTML}
                </div>
            </div>
        `;
    }

    async uploadTranscript(html, channelName) {
        try {
            const fileName = `transcript-${channelName}-${Date.now()}.html`;
            
            const formData = new FormData();
            const buffer = Buffer.from(html, 'utf8');
            formData.append('file', buffer, {
                filename: fileName,
                contentType: 'text/html'
            });

            const response = await axios.post(this.uploadApiUrl, formData, {
                headers: {
                    ...formData.getHeaders()
                }
            });

            if (response.data.success) {
                return {
                    url: response.data.data.url,
                    viewUrl: response.data.data.view_url
                };
            } else {
                throw new Error(response.data.message);
            }
        } catch (error) {
            console.error('Transcript yükleme hatası:', error);
            throw error;
        }
    }

    countAttachments(messages) {
        return messages.reduce((count, message) => count + message.attachments.size, 0);
    }

    formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
    }

    escapeHtml(text) {
        if (!text) return '';
        return text
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;')
            .replace(/"/g, '&quot;')
            .replace(/'/g, '&#39;');
    }
}

module.exports = TranscriptSistemi;