const { EmbedBuilder } = require('discord.js');
const config = require('../config.json');

class SpamKoruma {
    constructor(client = null) {
        this.client = client;
        this.aktif = config.SpamKoruma.aktif;
        this.esikDegeri = config.SpamKoruma.esikDegeri;
        this.istisnaliRoller = config.SpamKoruma.istisnaliRoller;
        this.uyariMesajlari = config.SpamKoruma.uyariMesajlari;
        this.spamlogchannelid = config.SpamKoruma.spamlogchannelid;
        this.yoneticiRolleri = config.SpamKoruma.yoneticiRolleri;
        
        this.kullaniciMesajlari = new Map();
        
        this.spamIstatistikleri = {
            toplamSpam: 0,
            bugunkuSpam: 0,
            sonSifirlanma: new Date().toDateString()
        };
        
        console.log('🛡️ Spam Koruma Sistemi başlatıldı');
    }
    
    /**
     * Sistemi aktif/pasif yapar
     * @param {boolean} durum - Sistem durumu
     */
    toggleSystem(durum) {
        this.aktif = durum;
        console.log(`🛡️ Spam Koruma Sistemi ${durum ? 'aktifleştirildi' : 'devre dışı bırakıldı'}`);
    }
    
    /**
     * Sistemin aktif olup olmadığını kontrol eder
     * @returns {boolean}
     */
    isActive() {
        return this.aktif;
    }
    
    /**
     * Kullanıcının spam yapıp yapmadığını kontrol eder
     * @param {Object} message - Discord mesaj objesi
     * @returns {boolean} - Spam tespit edildi mi?
     */
    async checkSpam(message) {
        if (!this.aktif) return false;
        
        const userId = message.author.id;
        const now = Date.now();
        
        if (message.author.bot) return false;
        
        if (this.hasExemptRole(message.member)) return false;
        
        if (!this.kullaniciMesajlari.has(userId)) {
            this.kullaniciMesajlari.set(userId, []);
        }
        
        const userMessages = this.kullaniciMesajlari.get(userId);
        
        const validMessages = userMessages.filter(timestamp => 
            now - timestamp < this.esikDegeri.zaman
        );
        
        validMessages.push(now);
        this.kullaniciMesajlari.set(userId, validMessages);
        
        if (validMessages.length > this.esikDegeri.mesajSayisi) {
            await this.handleSpam(message);
            return true;
        }
        
        return false;
    }
    
    /**
     * Kullanıcının istisnalı role sahip olup olmadığını kontrol eder
     * @param {Object} member - Discord üye objesi
     * @returns {boolean}
     */
    hasExemptRole(member) {
        if (!member || !member.roles) return false;
        
        return this.istisnaliRoller.some(roleId => 
            member.roles.cache.has(roleId)
        );
    }
    
    /**
     * Spam tespit edildiğinde yapılacak işlemler
     * @param {Object} message - Discord mesaj objesi
     */
    async handleSpam(message) {
        try {
            if (message.deletable) {
                await message.delete();
            }
            
            await this.sendDMWarning(message.author);
            await this.sendChannelWarning(message);
            await this.sendSpamLog(message);
            
            this.updateStats();
            
            console.log(`🚨 Spam tespit edildi: ${message.author.tag} (${message.author.id})`);
            
        } catch (error) {
            console.error('Spam işleme hatası:', error);
        }
    }
    
    /**
     * Kullanıcıya DM uyarısı gönderir
     * @param {Object} user - Discord kullanıcı objesi
     */
    async sendDMWarning(user) {
        try {
            const dmMessage = this.uyariMesajlari.dmUyari
                .replace('{kullanici}', user.username)
                .replace('{mesajSayisi}', this.esikDegeri.mesajSayisi)
                .replace('{zaman}', this.esikDegeri.zaman / 1000);
            
            const dmEmbed = new EmbedBuilder()
                .setColor(0xff9900)
                .setTitle('⚠️ Spam Uyarısı')
                .setDescription(dmMessage)
                .addFields(
                    {
                        name: '📊 Spam Limiti',
                        value: `• **${this.esikDegeri.mesajSayisi}** mesaj\n• **${this.esikDegeri.zaman / 1000}** saniye içinde`,
                        inline: true
                    },
                    {
                        name: '⚡ Öneri',
                        value: 'Mesajlarınız arasında daha fazla süre bırakın',
                        inline: true
                    }
                )
                .setFooter({ text: 'RoveBot\'s • Spam Koruma Sistemi' })
                .setTimestamp();
            
            await user.send({ embeds: [dmEmbed] });
            
        } catch (error) {
            console.log(`DM uyarısı gönderilemedi: ${user.tag} - ${error.message}`);
        }
    }
    
    /**
     * Kanala uyarı mesajı gönderir
     * @param {Object} message - Discord mesaj objesi
     */
    async sendChannelWarning(message) {
        try {
            const channelMessage = this.uyariMesajlari.kanalUyari
                .replace('{kullanici}', `<@${message.author.id}>`);
            
            const warningMessage = await message.channel.send(channelMessage);
            
            setTimeout(async () => {
                try {
                    if (warningMessage.deletable) {
                        await warningMessage.delete();
                    }
                } catch (error) {
                    console.log('Uyarı mesajı silinemedi:', error.message);
                }
            }, 5000);
            
        } catch (error) {
            console.error('Kanal uyarısı gönderilemedi:', error);
        }
    }
    
    /**
     * Spam log mesajı gönderir
     * @param {Object} message - Discord mesaj objesi
     */
    async sendSpamLog(message) {
        try {
            if (config.LogSistemi && config.LogSistemi.aktif && this.client && this.client.logSistemi) {
                const userMessages = this.kullaniciMesajlari.get(message.author.id) || [];
                await this.client.logSistemi.logSpam(
                    message.member, 
                    userMessages.length, 
                    this.esikDegeri.zaman
                );
                return;
            }

            if (this.spamlogchannelid === 'SPAM_LOG_KANAL_ID_BURAYA') return;
            
            const logChannel = message.guild.channels.cache.get(this.spamlogchannelid);
            if (!logChannel) return;
            
            const userMessages = this.kullaniciMesajlari.get(message.author.id) || [];
            
            const logEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('🚨 Spam Tespit Edildi')
                .setDescription(`**${message.author.tag}** spam yaptığı tespit edildi`)
                .addFields(
                    {
                        name: '👤 Kullanıcı Bilgileri',
                        value: `• **Kullanıcı:** <@${message.author.id}>\n• **ID:** \`${message.author.id}\`\n• **Tag:** ${message.author.tag}`,
                        inline: true
                    },
                    {
                        name: '📍 Konum Bilgileri',
                        value: `• **Kanal:** <#${message.channel.id}>\n• **Sunucu:** ${message.guild.name}\n• **Zaman:** <t:${Math.floor(Date.now() / 1000)}:F>`,
                        inline: true
                    },
                    {
                        name: '📊 Spam Detayları',
                        value: `• **Mesaj Sayısı:** ${userMessages.length}/${this.esikDegeri.mesajSayisi}\n• **Zaman Aralığı:** ${this.esikDegeri.zaman / 1000} saniye\n• **Durum:** Mesaj silindi, uyarı gönderildi`,
                        inline: false
                    }
                )
                .setThumbnail(message.author.displayAvatarURL())
                .setFooter({ text: `RoveBot\'s • Spam Koruma Sistemi` })
                .setTimestamp();
            
            await logChannel.send({ embeds: [logEmbed] });
            
        } catch (error) {
            console.error('Spam log gönderilemedi:', error);
        }
    }
    
    /**
     * Spam istatistiklerini günceller
     */
    updateStats() {
        const today = new Date().toDateString();
        
        if (this.spamIstatistikleri.sonSifirlanma !== today) {
            this.spamIstatistikleri.bugunkuSpam = 0;
            this.spamIstatistikleri.sonSifirlanma = today;
        }
        
        this.spamIstatistikleri.toplamSpam++;
        this.spamIstatistikleri.bugunkuSpam++;
    }
    
    /**
     * Kullanıcı spam istatistiklerini getirir
     * @param {string} userId - Kullanıcı ID'si
     * @returns {Object}
     */
    getUserStats(userId) {
        const userMessages = this.kullaniciMesajlari.get(userId) || [];
        const now = Date.now();
        
        const recentMessages = userMessages.filter(timestamp => 
            now - timestamp < 86400000
        );
        
        return {
            toplamMesaj: userMessages.length,
            son24SaatMesaj: recentMessages.length,
            sonMesajZamani: userMessages.length > 0 ? new Date(userMessages[userMessages.length - 1]) : null,
            aktifSpamDurumu: userMessages.length > this.esikDegeri.mesajSayisi
        };
    }
    
    /**
     * Sistem istatistiklerini getirir
     * @returns {Object}
     */
    getSystemStats() {
        return {
            sistemAktif: this.aktif,
            toplamKullanici: this.kullaniciMesajlari.size,
            toplamSpam: this.spamIstatistikleri.toplamSpam,
            bugunkuSpam: this.spamIstatistikleri.bugunkuSpam,
            esikDegeri: this.esikDegeri,
            istisnaliRolSayisi: this.istisnaliRoller.length
        };
    }
    
    /**
     * Kullanıcı mesaj geçmişini temizler
     * @param {string} userId - Kullanıcı ID'si (opsiyonel, belirtilmezse tümü temizlenir)
     */
    clearUserHistory(userId = null) {
        if (userId) {
            this.kullaniciMesajlari.delete(userId);
            console.log(`Kullanıcı geçmişi temizlendi: ${userId}`);
        } else {
            this.kullaniciMesajlari.clear();
            console.log('Tüm kullanıcı geçmişi temizlendi');
        }
    }
    
    /**
     * Eski mesaj kayıtlarını otomatik temizler (performans için)
     */
    cleanupOldRecords() {
        const now = Date.now();
        const cleanupThreshold = this.esikDegeri.zaman * 2;
        let cleanedUsers = 0;
        let cleanedMessages = 0;
        
        for (const [userId, messages] of this.kullaniciMesajlari.entries()) {
            const originalCount = messages.length;
            const validMessages = messages.filter(timestamp => 
                now - timestamp < cleanupThreshold
            );
            
            if (validMessages.length === 0) {
                this.kullaniciMesajlari.delete(userId);
                cleanedUsers++;
                cleanedMessages += originalCount;
            } else if (validMessages.length < originalCount) {
                this.kullaniciMesajlari.set(userId, validMessages);
                cleanedMessages += (originalCount - validMessages.length);
            }
        }
        
        // Aggressive cleanup if memory usage is high
        if (this.kullaniciMesajlari.size > 1000) {
            console.warn('⚠️ Spam koruma bellek kullanımı yüksek, agresif temizlik yapılıyor...');
            
            // Keep only the most recent records for users with many messages
            for (const [userId, messages] of this.kullaniciMesajlari.entries()) {
                if (messages.length > 10) {
                    const recentMessages = messages.slice(-5); // Keep only last 5 messages
                    this.kullaniciMesajlari.set(userId, recentMessages);
                    cleanedMessages += (messages.length - recentMessages.length);
                }
            }
        }
        
        if (cleanedUsers > 0 || cleanedMessages > 0) {
            console.log(`🧹 Spam koruma temizliği: ${cleanedUsers} kullanıcı, ${cleanedMessages} mesaj kaydı temizlendi`);
        }
        
        // Log memory usage
        const memoryUsage = this.getMemoryUsage();
        if (memoryUsage.users > 500) {
            console.warn(`📊 Spam koruma bellek kullanımı: ${memoryUsage.users} kullanıcı, ${memoryUsage.totalMessages} mesaj`);
        }
    }
    
    /**
     * Get current memory usage statistics
     * @returns {Object} Memory usage stats
     */
    getMemoryUsage() {
        let totalMessages = 0;
        for (const messages of this.kullaniciMesajlari.values()) {
            totalMessages += messages.length;
        }
        
        return {
            users: this.kullaniciMesajlari.size,
            totalMessages: totalMessages,
            averageMessagesPerUser: this.kullaniciMesajlari.size > 0 ? Math.round(totalMessages / this.kullaniciMesajlari.size) : 0
        };
    }
}

module.exports = SpamKoruma;