const { EmbedBuilder, AuditLogEvent } = require('discord.js');
const fs = require('fs');
const path = require('path');

class LogSistemi {
    constructor(client) {
        this.client = client;
        this.config = this.loadConfig();
    }

    loadConfig() {
        try {
            const configPath = path.join(__dirname, '../config.json');
            return JSON.parse(fs.readFileSync(configPath, 'utf8'));
        } catch (error) {
            console.error('Config dosyası yüklenirken hata:', error);
            return null;
        }
    }

    async sendLog(logType, embed) {
        try {
            if (!this.config || !this.config.LogSistemi.aktif) {
                return false;
            }

            const channelId = this.config.LogSistemi.logKanallari[logType];
            if (!channelId) {
                console.log(`${logType} için kanal ID bulunamadı`);
                await this.notifyAdministrators(`Log kanalı bulunamadı: ${logType}`);
                return false;
            }

            const channel = this.client.channels.cache.get(channelId);
            if (!channel) {
                console.log(`${logType} kanalı bulunamadı: ${channelId}`);
                await this.notifyAdministrators(`Log kanalı erişilemez: ${logType} (${channelId})`);
                return false;
            }

            await channel.send({ embeds: [embed] });
            return true;
        } catch (error) {
            console.error(`Log gönderme hatası (${logType}):`, error);
            await this.notifyAdministrators(`Log gönderme hatası: ${logType} - ${error.message}`);
            return false;
        }
    }

    async logBackupOperation(guildId, operation, userId, status, details = '') {
        try {
            const config = this.loadConfig();
            if (!config || !config.LogSistemi || !config.LogSistemi.logKanallari || !config.LogSistemi.logKanallari.backupLog) {
                console.log('Backup log kanalı bulunamadı');
                return;
            }

            const guild = this.client.guilds.cache.get(guildId);
            if (!guild) return;

            const logChannel = guild.channels.cache.get(config.LogSistemi.logKanallari.backupLog);
            if (!logChannel) return;

            const user = await this.client.users.fetch(userId).catch(() => null);
            const timestamp = new Date().toLocaleString('tr-TR', { timeZone: 'Europe/Istanbul' });

            let color;
            switch (status) {
                case 'BAŞARILI':
                    color = '#00ff00';
                    break;
                case 'HATA':
                    color = '#ff0000';
                    break;
                case 'BAŞLADI':
                    color = '#ffaa00';
                    break;
                default:
                    color = '#0099ff';
            }

            const embed = new EmbedBuilder()
                .setTitle(`💾 Yedekleme Sistemi - ${operation}`)
                .setColor(color)
                .addFields(
                    { name: '👤 Kullanıcı', value: user ? `${user.tag} (${user.id})` : `ID: ${userId}`, inline: true },
                    { name: '🏷️ İşlem', value: operation, inline: true },
                    { name: '📊 Durum', value: status, inline: true },
                    { name: '🕐 Zaman', value: timestamp, inline: false }
                )
                .setTimestamp();

            if (details) {
                embed.addFields({ name: '📝 Detaylar', value: details, inline: false });
            }

            await logChannel.send({ embeds: [embed] });

            const logEntry = `[${timestamp}] [${operation}] [${user ? user.tag : userId}] [${status}] ${details}\n`;
            const logFilePath = path.join(__dirname, '../logs/backup_system.log');
            
            const logsDir = path.dirname(logFilePath);
            if (!fs.existsSync(logsDir)) {
                fs.mkdirSync(logsDir, { recursive: true });
            }

            fs.appendFileSync(logFilePath, logEntry, 'utf8');

        } catch (error) {
            console.error('Backup log yazılırken hata:', error);
        }
    }

    async logBackupStart(guildId, userId, backupType = 'MANUEL') {
        await this.logBackupOperation(guildId, 'YEDEKLEME', userId, 'BAŞLADI', `${backupType} yedekleme işlemi başlatıldı`);
    }

    async logBackupSuccess(guildId, userId, fileName, stats) {
        const details = `Dosya: ${fileName}\nKanallar: ${stats.channels || 0}\nRoller: ${stats.roles || 0}\nKullanıcı Rolleri: ${stats.userRoles || 0}\nEmojiler: ${stats.emojis || 0}`;
        await this.logBackupOperation(guildId, 'YEDEKLEME', userId, 'BAŞARILI', details);
    }

    async logBackupError(guildId, userId, error) {
        await this.logBackupOperation(guildId, 'YEDEKLEME', userId, 'HATA', `Hata: ${error}`);
    }

    async logRestoreStart(guildId, userId, fileName) {
        await this.logBackupOperation(guildId, 'GERİ YÜKLEME', userId, 'BAŞLADI', `Dosya: ${fileName}`);
    }

    async logRestoreSuccess(guildId, userId, fileName, stats) {
        const details = `Dosya: ${fileName}\nGeri Yüklenen Kanallar: ${stats.channels || 0}\nGeri Yüklenen Roller: ${stats.roles || 0}\nAtanan Kullanıcı Rolleri: ${stats.userRoles || 0}\nGeri Yüklenen Emojiler: ${stats.emojis || 0}`;
        await this.logBackupOperation(guildId, 'GERİ YÜKLEME', userId, 'BAŞARILI', details);
    }

    async logRestoreError(guildId, userId, fileName, error) {
        await this.logBackupOperation(guildId, 'GERİ YÜKLEME', userId, 'HATA', `Dosya: ${fileName}\nHata: ${error}`);
    }

    async logBackupValidation(guildId, userId, fileName, isValid, reason = '') {
        const status = isValid ? 'GEÇERLİ' : 'GEÇERSİZ';
        const details = `Dosya: ${fileName}${reason ? `\nSebep: ${reason}` : ''}`;
        await this.logBackupOperation(guildId, 'DOSYA DOĞRULAMA', userId, status, details);
    }

    async notifyAdministrators(errorMessage) {
        try {
            if (!this.config?.LogSistemi?.adminNotifications?.enabled) return;
            
            const adminChannelId = this.config.LogSistemi.adminNotifications.channelId;
            if (!adminChannelId) return;

            const adminChannel = this.client.channels.cache.get(adminChannelId);
            if (!adminChannel) return;

            const embed = new EmbedBuilder()
                .setTitle('⚠️ Log Sistemi Hatası')
                .setDescription(`**Hata:** ${errorMessage}`)
                .addFields({
                    name: '🕐 Hata Zamanı',
                    value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                    inline: true
                })
                .setColor('#ff0000')
                .setTimestamp();

            await adminChannel.send({ embeds: [embed] });
        } catch (error) {
            console.error('Admin bildirim hatası:', error);
        }
    }

    async getAuditLogEntry(guild, actionType, targetId, timeLimit = 5000) {
        try {
            const botMember = guild.members.cache.get(guild.client.user.id);
            if (!botMember || !botMember.permissions.has('ViewAuditLog')) {
                console.warn('Bot audit log okuma yetkisine sahip değil');
                return null;
            }

            const auditLogs = await guild.fetchAuditLogs({
                type: actionType,
                limit: 10
            });

            console.log(`Audit log sorgusu: ${actionType}, Target: ${targetId}, Bulunan entry sayısı: ${auditLogs.entries.size}`);

            const entry = auditLogs.entries.find(entry => {
                const timeDiff = Date.now() - entry.createdTimestamp;
                const isTargetMatch = entry.target?.id === targetId;
                const isTimeValid = timeDiff < timeLimit;
                
                console.log(`Entry kontrol: Target match: ${isTargetMatch}, Time diff: ${timeDiff}ms, Valid: ${isTimeValid}`);
                
                return isTargetMatch && isTimeValid;
            });

            if (entry) {
                console.log(`Audit log entry bulundu: ${entry.executor?.tag} tarafından ${new Date(entry.createdTimestamp).toLocaleString()}`);
            }

            return entry;
        } catch (error) {
            console.error('Audit log alınırken hata:', error);
            return null;
        }
    }

    async logMemberJoin(member) {
        const embed = new EmbedBuilder()
            .setTitle('💻 Yeni Üye Katıldı')
            .setDescription(`**Kullanıcı:** ${member.user.tag} (${member.user.id})\n**Mention:** <@${member.user.id}>`)
            .addFields(
                { name: '📅 Hesap Oluşturma', value: `<t:${Math.floor(member.user.createdTimestamp / 1000)}:F>`, inline: true },
                { name: '🕐 Katılma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true },
                { name: '👥 Toplam Üye', value: `${member.guild.memberCount}`, inline: true }
            )
            .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
            .setColor('#00ff00')
            .setTimestamp();

        await this.sendLog('girisLog', embed);
        await this.checkSuspiciousAccount(member);
    }

    async logMemberLeave(member) {
        const embed = new EmbedBuilder()
            .setTitle('🚪 Üye Ayrıldı')
            .setDescription(`**Kullanıcı:** ${member.user.tag} (${member.user.id})\n**Mention:** <@${member.user.id}>`)
            .addFields(
                { name: '📅 Hesap Oluşturma', value: `<t:${Math.floor(member.user.createdTimestamp / 1000)}:F>`, inline: true },
                { name: '🕐 Ayrılma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true },
                { name: '👥 Kalan Üye', value: `${member.guild.memberCount}`, inline: true }
            )
            .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
            .setColor('#ff0000')
            .setTimestamp();

        await this.sendLog('cikisLog', embed);
    }

    async logRoleUpdate(oldMember, newMember) {
        const addedRoles = newMember.roles.cache.filter(role => !oldMember.roles.cache.has(role.id));
        const removedRoles = oldMember.roles.cache.filter(role => !newMember.roles.cache.has(role.id));

        if (addedRoles.size === 0 && removedRoles.size === 0) return;

        let executor = null;
        try {
            const auditEntry = await this.getAuditLogEntry(
                newMember.guild, 
                AuditLogEvent.MemberRoleUpdate, 
                newMember.user.id
            );
            executor = auditEntry?.executor;
        } catch (error) {
            console.error('Rol değişikliği audit log hatası:', error);
        }

        const embed = new EmbedBuilder()
            .setTitle('🎭 Rol Değişikliği')
            .setDescription(`**Kullanıcı:** ${newMember.user.tag} (${newMember.user.id})\n**Mention:** <@${newMember.user.id}>`)
            .setThumbnail(newMember.user.displayAvatarURL({ dynamic: true }))
            .setColor('#ffff00')
            .setTimestamp();

        if (addedRoles.size > 0) {
            embed.addFields({
                name: '➕ Eklenen Roller',
                value: addedRoles.map(role => `<@&${role.id}> (${role.name})`).join('\n'),
                inline: false
            });
        }

        if (removedRoles.size > 0) {
            embed.addFields({
                name: '➖ Kaldırılan Roller',
                value: removedRoles.map(role => `<@&${role.id}> (${role.name})`).join('\n'),
                inline: false
            });
        }

        if (executor) {
            embed.addFields({
                name: '👮 İşlemi Yapan Yetkili',
                value: `${executor.tag} (${executor.id})\n<@${executor.id}>`,
                inline: true
            });
        }

        embed.addFields({
            name: '🕐 İşlem Zamanı',
            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
            inline: true
        });

        await this.sendLog('rolLog', embed);
    }

    async logBan(ban) {
        let executor = null;
        let auditReason = ban.reason;
        
        try {
            const auditEntry = await this.getAuditLogEntry(
                ban.guild, 
                AuditLogEvent.MemberBanAdd, 
                ban.user.id
            );
            
            if (auditEntry) {
                executor = auditEntry.executor;
                auditReason = auditEntry.reason || ban.reason;
            }
        } catch (error) {
            console.error('Ban audit log hatası:', error);
        }

        const embed = new EmbedBuilder()
            .setTitle('🔨 Üye Banlandı')
            .setDescription(`**Kullanıcı:** ${ban.user.tag} (${ban.user.id})\n**Mention:** <@${ban.user.id}>`)
            .addFields(
                { name: '📝 Ban Sebebi', value: auditReason || 'Sebep belirtilmemiş', inline: false },
                { name: '🕐 Ban Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(ban.user.displayAvatarURL({ dynamic: true }))
            .setColor('#8B0000')
            .setTimestamp();

        if (executor) {
            embed.addFields({
                name: '👮 Ban İşlemini Yapan Yetkili',
                value: `${executor.tag} (${executor.id})\n<@${executor.id}>`,
                inline: true
            });
        }

        const member = ban.guild.members.cache.get(ban.user.id);
        if (member && member.roles.cache.size > 1) {
            const roles = member.roles.cache
                .filter(role => role.id !== ban.guild.id)
                .map(role => `<@&${role.id}>`)
                .join(', ');
            
            if (roles.length > 0) {
                embed.addFields({
                    name: '🎭 Kullanıcının Rolleri',
                    value: roles.length > 1000 ? roles.substring(0, 1000) + '...' : roles,
                    inline: false
                });
            }
        }

        await this.sendLog('banLog', embed);
    }

    async logKick(member, reason, executor) {
        if (!executor) {
            try {
                const auditEntry = await this.getAuditLogEntry(
                    member.guild, 
                    AuditLogEvent.MemberKick, 
                    member.user.id
                );
                executor = auditEntry?.executor;
                reason = auditEntry?.reason || reason;
            } catch (error) {
                console.error('Kick audit log hatası:', error);
            }
        }

        const embed = new EmbedBuilder()
            .setTitle('👢 Üye Atıldı')
            .setDescription(`**Kullanıcı:** ${member.user.tag} (${member.user.id})\n**Mention:** <@${member.user.id}>`)
            .addFields(
                { name: '📝 Atılma Sebebi', value: reason || 'Sebep belirtilmemiş', inline: false },
                { name: '🕐 Atılma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
            .setColor('#FF4500')
            .setTimestamp();

        if (executor) {
            embed.addFields({
                name: '👮 Atma İşlemini Yapan Yetkili',
                value: `${executor.tag} (${executor.id})\n<@${executor.id}>`,
                inline: true
            });
        }

        if (member.roles.cache.size > 1) {
            const roles = member.roles.cache
                .filter(role => role.id !== member.guild.id)
                .map(role => `<@&${role.id}>`)
                .join(', ');
            
            if (roles.length > 0) {
                embed.addFields({
                    name: '🎭 Kullanıcının Rolleri',
                    value: roles.length > 1000 ? roles.substring(0, 1000) + '...' : roles,
                    inline: false
                });
            }
        }

        if (member.joinedAt) {
            embed.addFields({
                name: '📅 Sunucuya Katılma Tarihi',
                value: `<t:${Math.floor(member.joinedAt.getTime() / 1000)}:F>`,
                inline: true
            });
        }

        await this.sendLog('kickLog', embed);
    }

    async logMute(member, duration, reason, executor) {
        if (!executor) {
            try {
                const auditEntry = await this.getAuditLogEntry(
                    member.guild, 
                    AuditLogEvent.MemberUpdate, 
                    member.user.id
                );
                executor = auditEntry?.executor;
                reason = auditEntry?.reason || reason;
            } catch (error) {
                console.error('Mute audit log hatası:', error);
            }
        }

        const embed = new EmbedBuilder()
            .setTitle('🔇 Üye Susturuldu')
            .setDescription(`**Kullanıcı:** ${member.user.tag} (${member.user.id})\n**Mention:** <@${member.user.id}>`)
            .addFields(
                { name: '📝 Susturma Sebebi', value: reason || 'Sebep belirtilmemiş', inline: false },
                { name: '⏰ Süre', value: duration ? `${Math.floor(duration / 60000)} dakika` : 'Süresiz', inline: true },
                { name: '🕐 Susturma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
            .setColor('#808080')
            .setTimestamp();

        if (executor) {
            embed.addFields({
                name: '👮 Susturma İşlemini Yapan Yetkili',
                value: `${executor.tag} (${executor.id})\n<@${executor.id}>`,
                inline: true
            });
        }

        await this.sendLog('muteLog', embed);
    }

    
    async logUnmute(member, executor) {
        if (!executor) {
            try {
                const auditEntry = await this.getAuditLogEntry(
                    member.guild, 
                    AuditLogEvent.MemberUpdate, 
                    member.user.id
                );
                executor = auditEntry?.executor;
            } catch (error) {
                console.error('Unmute audit log hatası:', error);
            }
        }

        const embed = new EmbedBuilder()
            .setTitle('🔊 Üye Susturması Kaldırıldı')
            .setDescription(`**Kullanıcı:** ${member.user.tag} (${member.user.id})\n**Mention:** <@${member.user.id}>`)
            .addFields(
                { name: '🕐 Kaldırılma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
            .setColor('#00ff00')
            .setTimestamp();

        if (executor) {
            embed.addFields({
                name: '👮 Kaldırma İşlemini Yapan Yetkili',
                value: `${executor.tag} (${executor.id})\n<@${executor.id}>`,
                inline: true
            });
        } else {
            embed.addFields({
                name: '🔄 Kaldırılma Türü',
                value: 'Otomatik (süre doldu)',
                inline: true
            });
        }

        await this.sendLog('muteLog', embed);
    }

    async logTimeout(member, duration, reason, executor) {
        if (!executor) {
            try {
                const auditEntry = await this.getAuditLogEntry(
                    member.guild, 
                    AuditLogEvent.MemberUpdate, 
                    member.user.id
                );
                executor = auditEntry?.executor;
                reason = auditEntry?.reason || reason;
            } catch (error) {
                console.error('Timeout audit log hatası:', error);
            }
        }

        const embed = new EmbedBuilder()
            .setTitle('⏰ Üye Timeout Aldı')
            .setDescription(`**Kullanıcı:** ${member.user.tag} (${member.user.id})\n**Mention:** <@${member.user.id}>`)
            .addFields(
                { name: '📝 Timeout Sebebi', value: reason || 'Sebep belirtilmemiş', inline: false },
                { name: '⏰ Süre', value: duration ? `${Math.floor(duration / 60000)} dakika` : 'Süresiz', inline: true },
                { name: '🕐 Timeout Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
            .setColor('#ff6600')
            .setTimestamp();

        if (duration && duration > 0) {
            const endTime = Math.floor((Date.now() + duration) / 1000);
            embed.addFields({
                name: '🔚 Timeout Bitiş Zamanı',
                value: `<t:${endTime}:F> (<t:${endTime}:R>)`,
                inline: true
            });
        }

        if (executor) {
            embed.addFields({
                name: '👮 Timeout İşlemini Yapan Yetkili',
                value: `${executor.tag} (${executor.id})\n<@${executor.id}>`,
                inline: true
            });
        }

        await this.sendLog('timeoutLog', embed);
    }

    async logTimeoutRemove(member, executor) {
        if (!executor) {
            try {
                const auditEntry = await this.getAuditLogEntry(
                    member.guild, 
                    AuditLogEvent.MemberUpdate, 
                    member.user.id
                );
                executor = auditEntry?.executor;
            } catch (error) {
                console.error('Timeout kaldırma audit log hatası:', error);
            }
        }

        const embed = new EmbedBuilder()
            .setTitle('✅ Timeout Kaldırıldı')
            .setDescription(`**Kullanıcı:** ${member.user.tag} (${member.user.id})\n**Mention:** <@${member.user.id}>`)
            .addFields(
                { name: '🕐 Kaldırılma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
            .setColor('#00ff00')
            .setTimestamp();

        if (executor) {
            embed.addFields({
                name: '👮 Kaldırma İşlemini Yapan Yetkili',
                value: `${executor.tag} (${executor.id})\n<@${executor.id}>`,
                inline: true
            });
        } else {
            embed.addFields({
                name: '🔄 Kaldırılma Türü',
                value: 'Otomatik (süre doldu)',
                inline: true
            });
        }

        await this.sendLog('timeoutLog', embed);
    }

    async logMessageDelete(message) {
        if (!message || !message.author) {
            console.error('LogSistemi.logMessageDelete: Geçersiz message objesi');
            return;
        }

        if (message.author.bot) return;

        let deleter = null;
        try {
            const auditEntry = await this.getAuditLogEntry(
                message.guild, 
                AuditLogEvent.MessageDelete, 
                message.author.id,
                5000
            );
            
            if (auditEntry) {
                deleter = auditEntry.executor;
                console.log(`Mesaj silme audit log bulundu: ${deleter?.tag} tarafından silindi`);
            } else {
                console.log('Mesaj silme audit log bulunamadı - kullanıcı kendisi silmiş olabilir');
            }
        } catch (error) {
            console.error('Mesaj silme audit log hatası:', error);
        }

        const embed = new EmbedBuilder()
            .setTitle('💬 Mesaj Silindi')
            .setDescription(`**Mesaj Sahibi:** ${message.author.tag} (${message.author.id})\n**Kanal:** <#${message.channel.id}>`)
            .addFields(
                { name: '📝 Silinen Mesaj', value: `\`\`\`${message.content && message.content.length > 500 ? message.content.substring(0, 500) + '...' : message.content || 'İçerik bulunamadı'}\`\`\``, inline: false },
                { name: '🕐 Silme Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(message.author.displayAvatarURL({ dynamic: true }))
            .setColor('#ff0000')
            .setTimestamp();

        if (deleter && deleter.id !== message.author.id) {
            embed.addFields({
                name: '👮 Mesajı Silen Yetkili',
                value: `${deleter.tag} (${deleter.id})\n<@${deleter.id}>`,
                inline: true
            });
            embed.setColor('#ff4500');
        } else {
            embed.addFields({
                name: '🗑️ Silme Türü',
                value: 'Kullanıcı tarafından silindi',
                inline: true
            });
        }

        if (message.attachments.size > 0) {
            const attachments = message.attachments.map(att => `[${att.name}](${att.url})`).join('\n');
            embed.addFields({
                name: '📎 Ekler',
                value: attachments.length > 1000 ? attachments.substring(0, 1000) + '...' : attachments,
                inline: false
            });
        }

        await this.sendLog('mesajLog', embed);
    }

    async logMessageUpdate(oldMessage, newMessage) {
        if (!oldMessage || !newMessage || !newMessage.author) {
            console.error('LogSistemi.logMessageUpdate: Geçersiz message objeleri');
            return;
        }

        if (newMessage.author.bot) return;
        if (oldMessage.content === newMessage.content) return;

        const embed = new EmbedBuilder()
            .setTitle('💬 Mesaj Düzenlendi')
            .setDescription(`**Kullanıcı:** ${newMessage.author.tag} (${newMessage.author.id})\n**Kanal:** <#${newMessage.channel.id}>\n**Mesaj Linki:** [Mesaja Git](${newMessage.url})`)
            .addFields(
                { name: '📝 Eski Mesaj', value: `\`\`\`${oldMessage.content && oldMessage.content.length > 250 ? oldMessage.content.substring(0, 250) + '...' : oldMessage.content || 'İçerik bulunamadı'}\`\`\``, inline: false },
                { name: '📝 Yeni Mesaj', value: `\`\`\`${newMessage.content && newMessage.content.length > 250 ? newMessage.content.substring(0, 250) + '...' : newMessage.content || 'İçerik bulunamadı'}\`\`\``, inline: false },
                { name: '🕐 Düzenleme Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(newMessage.author.displayAvatarURL({ dynamic: true }))
            .setColor('#ffff00')
            .setTimestamp();

        await this.sendLog('mesajLog', embed);
    }

    async checkSuspiciousAccount(member) {
        const sixMonthsAgo = Date.now() - (6 * 30 * 24 * 60 * 60 * 1000);
        const accountAge = member.user.createdTimestamp;

        if (accountAge > sixMonthsAgo) {
            const daysDiff = Math.floor((Date.now() - accountAge) / (1000 * 60 * 60 * 24));
            
            const embed = new EmbedBuilder()
                .setTitle('🔍 Şüpheli Hesap Tespit Edildi')
                .setDescription(`**Kullanıcı:** ${member.user.tag} (${member.user.id})\n**Mention:** <@${member.user.id}>`)
                .addFields(
                    { name: '📅 Hesap Yaşı', value: `${daysDiff} gün`, inline: true },
                    { name: '⚠️ Risk Seviyesi', value: daysDiff < 7 ? '🔴 Yüksek' : daysDiff < 30 ? '🟡 Orta' : '🟢 Düşük', inline: true },
                    { name: '🕐 Katılma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
                )
                .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
                .setColor(daysDiff < 7 ? '#ff0000' : daysDiff < 30 ? '#ffff00' : '#00ff00')
                .setTimestamp();

            await this.sendLog('suuheliLog', embed);
        }
    }

    async logInviteCreate(invite) {
        const embed = new EmbedBuilder()
            .setTitle('🎫 Yeni Davet Oluşturuldu')
            .setDescription(`**Davet Kodu:** \`${invite.code}\`\n**Kanal:** <#${invite.channel.id}>`)
            .addFields(
                { name: '👤 Oluşturan', value: invite.inviter ? `${invite.inviter.tag} (${invite.inviter.id})\n<@${invite.inviter.id}>` : 'Bilinmiyor', inline: true },
                { name: '📊 Maksimum Kullanım', value: invite.maxUses === 0 ? 'Sınırsız' : invite.maxUses.toString(), inline: true },
                { name: '⏰ Süre', value: invite.maxAge === 0 ? 'Süresiz' : `${Math.floor(invite.maxAge / 3600)} saat`, inline: true },
                { name: '🕐 Oluşturma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setColor('#00ff00')
            .setTimestamp();

        await this.sendLog('davetLog', embed);
    }

    async logInviteDelete(invite) {
        const embed = new EmbedBuilder()
            .setTitle('🎫 Davet Silindi')
            .setDescription(`**Davet Kodu:** \`${invite.code}\`\n**Kanal:** <#${invite.channel.id}>`)
            .addFields(
                { name: '👤 Oluşturan', value: invite.inviter ? `${invite.inviter.tag} (${invite.inviter.id})\n<@${invite.inviter.id}>` : 'Bilinmiyor', inline: true },
                { name: '📊 Kullanım Sayısı', value: invite.uses?.toString() || '0', inline: true },
                { name: '🕐 Silinme Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setColor('#ff0000')
            .setTimestamp();

        await this.sendLog('davetLog', embed);
    }

    async logInviteUsed(invite, member) {
        const embed = new EmbedBuilder()
            .setTitle('🎫 Davet Kullanıldı')
            .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
            .setColor('#00aaff')
            .setTimestamp();

        if (invite.isVanity) {
            embed.setDescription(`**Davet Türü:** Vanity URL\n**Davet Kodu:** \`${invite.code}\``);
        } else if (invite.isUnknown) {
            embed.setDescription(`**Davet Türü:** Bilinmiyor (Bot davet, Widget vb.)\n**Davet Kodu:** \`${invite.code}\``);
        } else {
            embed.setDescription(`**Davet Kodu:** \`${invite.code}\`\n**Kanal:** ${invite.channel ? `<#${invite.channel.id}>` : 'Bilinmiyor'}`);
        }

        embed.addFields({
            name: '🆕 Katılan Üye',
            value: `${member.user.tag} (${member.user.id})\n<@${member.user.id}>`,
            inline: true
        });

        if (invite.inviter && !invite.isVanity && !invite.isUnknown) {
            embed.addFields({
                name: '👤 Davet Sahibi',
                value: `${invite.inviter.tag} (${invite.inviter.id})\n<@${invite.inviter.id}>`,
                inline: true
            });

            if (member.client.inviteTracker) {
                const stats = member.client.inviteTracker.getInviteStats(member.guild.id, invite.inviter.id);
                if (stats) {
                    embed.addFields({
                        name: '📈 Davet Sahibi İstatistikleri',
                        value: `**Toplam Davet:** ${stats.totalInvites}\n**Aktif Davet Kodu:** ${stats.activeInvites}`,
                        inline: true
                    });
                }
            }
        } else {
            embed.addFields({
                name: '👤 Davet Sahibi',
                value: invite.isVanity ? 'Vanity URL (Sunucu)' : 'Bilinmiyor',
                inline: true
            });
        }

        if (!invite.isUnknown) {
            const currentUses = invite.uses || 0;
            const maxUses = invite.maxUses || 0;
            const remainingUses = maxUses === 0 ? 'Sınırsız' : Math.max(0, maxUses - currentUses);

            embed.addFields(
                {
                    name: '📊 Kullanım Sayısı',
                    value: `**Mevcut:** ${currentUses}\n**Maksimum:** ${maxUses === 0 ? 'Sınırsız' : maxUses}`,
                    inline: true
                },
                {
                    name: '📊 Kalan Kullanım',
                    value: remainingUses.toString(),
                    inline: true
                }
            );
        }

        const accountAge = Date.now() - member.user.createdAt.getTime();
        const daysSinceCreation = Math.floor(accountAge / (1000 * 60 * 60 * 24));
        
        embed.addFields({
            name: '📅 Hesap Bilgileri',
            value: `**Oluşturulma:** <t:${Math.floor(member.user.createdAt.getTime() / 1000)}:R>\n**Yaş:** ${daysSinceCreation} gün`,
            inline: true
        });

        if (daysSinceCreation < 7) {
            embed.addFields({
                name: '⚠️ Uyarı',
                value: 'Bu hesap 7 günden yeni!',
                inline: true
            });
            embed.setColor('#ff9900');
        }

        embed.addFields({
            name: '🕐 Katılım Zamanı',
            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
            inline: false
        });

        await this.sendLog('davetLog', embed);
    }

    async logSpam(member, messageCount, timeframe) {
        if (!member || !member.user) {
            console.error('LogSistemi.logSpam: Geçersiz member objesi');
            return;
        }

        const embed = new EmbedBuilder()
            .setTitle('🚫 Spam Tespit Edildi')
            .setDescription(`**Kullanıcı:** ${member.user.tag} (${member.user.id})\n**Mention:** <@${member.user.id}>`)
            .addFields(
                { name: '📊 Mesaj Sayısı', value: messageCount.toString(), inline: true },
                { name: '⏰ Zaman Aralığı', value: `${timeframe / 1000} saniye`, inline: true },
                { name: '🕐 Tespit Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
            .setColor('#ff6600')
            .setTimestamp();

        await this.sendLog('spamLog', embed);
    }

    async logProfanity(member, messageContent, profanityType, timeoutDuration) {
        if (!member || !member.user) {
            console.error('LogSistemi.logProfanity: Geçersiz member objesi');
            return;
        }

        const embed = new EmbedBuilder()
            .setTitle('🤬 Küfür Tespit Edildi')
            .setDescription(`**Kullanıcı:** ${member.user.tag} (${member.user.id})\n**Mention:** <@${member.user.id}>`)
            .addFields(
                { name: '📝 Mesaj İçeriği', value: `\`\`\`${messageContent.length > 100 ? messageContent.substring(0, 100) + '...' : messageContent}\`\`\``, inline: false },
                { name: '🏷️ Küfür Türü', value: profanityType, inline: true },
                { name: '⏰ Timeout Süresi', value: timeoutDuration > 0 ? `${Math.floor(timeoutDuration / 60000)} dakika` : 'Uyarı', inline: true },
                { name: '🕐 Tespit Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
            .setColor('#8B0000')
            .setTimestamp();

        await this.sendLog('kufurLog', embed);
    }

    async logMessageUpdate(oldMessage, newMessage) {
        if (!oldMessage || !newMessage || !newMessage.author) {
            console.error('LogSistemi.logMessageUpdate: Geçersiz message objeleri');
            return;
        }

        if (newMessage.author.bot) return;
        if (oldMessage.content === newMessage.content) return;

        const embed = new EmbedBuilder()
            .setTitle('💬 Mesaj Düzenlendi')
            .setDescription(`**Kullanıcı:** ${newMessage.author.tag} (${newMessage.author.id})\n**Kanal:** <#${newMessage.channel.id}>\n**Mesaj Linki:** [Mesaja Git](${newMessage.url})`)
            .addFields(
                { name: '📝 Eski Mesaj', value: `\`\`\`${oldMessage.content && oldMessage.content.length > 250 ? oldMessage.content.substring(0, 250) + '...' : oldMessage.content || 'İçerik bulunamadı'}\`\`\``, inline: false },
                { name: '📝 Yeni Mesaj', value: `\`\`\`${newMessage.content && newMessage.content.length > 250 ? newMessage.content.substring(0, 250) + '...' : newMessage.content || 'İçerik bulunamadı'}\`\`\``, inline: false },
                { name: '🕐 Düzenleme Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(newMessage.author.displayAvatarURL({ dynamic: true }))
            .setColor('#ffff00')
            .setTimestamp();

        await this.sendLog('mesajLog', embed);
    }

    async logTicketOpen(guild, ticketData) {
        if (!guild || !ticketData || !ticketData.user || !ticketData.channel) {
            console.error('LogSistemi.logTicketOpen: Geçersiz parametreler');
            return;
        }

        const { user, channel: ticketChannel, category, categoryEmoji } = ticketData;

        const embed = new EmbedBuilder()
            .setTitle('🎟️ Yeni Ticket Açıldı')
            .setDescription(`**Kullanıcı:** ${user.tag} (${user.id})\n**Mention:** <@${user.id}>`)
            .addFields(
                { name: '📂 Ticket Kanalı', value: `<#${ticketChannel.id}>`, inline: true },
                { name: '🏷️ Kategori', value: `${categoryEmoji || '🎫'} ${category || 'Belirtilmemiş'}`, inline: true },
                { name: '🆔 Kanal ID', value: ticketChannel.id, inline: true },
                { name: '🕐 Açılma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: false }
            )
            .setThumbnail(user.displayAvatarURL({ dynamic: true }))
            .setColor('#00ff00')
            .setTimestamp();

        await this.sendLog('ticketAcmaLog', embed);
    }

    async logTicketClose(guild, ticketData) {
        if (!guild || !ticketData || !ticketData.channel || !ticketData.closedBy) {
            console.error('LogSistemi.logTicketClose: Geçersiz parametreler');
            return;
        }

        const { channel: ticketChannel, closedBy, ticketOwner, transcriptUrl } = ticketData;

        const embed = new EmbedBuilder()
            .setTitle('🔒 Ticket Kapatıldı')
            .setDescription(`**Ticket Sahibi:** ${ticketOwner ? `${ticketOwner.user.tag} (${ticketOwner.user.id})` : 'Bilinmiyor'}\n**Kapatan:** ${closedBy.tag} (${closedBy.id})`)
            .addFields(
                { name: '📂 Kapatılan Kanal', value: `#${ticketChannel.name}`, inline: true },
                { name: '🆔 Kanal ID', value: ticketChannel.id, inline: true },
                { name: '🕐 Kapatılma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: false }
            )
            .setThumbnail(ticketOwner ? ticketOwner.user.displayAvatarURL({ dynamic: true }) : closedBy.displayAvatarURL({ dynamic: true }))
            .setColor('#ff0000')
            .setTimestamp();

        if (transcriptUrl) {
            embed.addFields({ name: '📄 Transcript', value: `[Konuşma Geçmişini Görüntüle](${transcriptUrl})`, inline: false });
        }

        await this.sendLog('ticketKapamaLog', embed);
    }

    async logTicketTranscript(guild, ticketData) {
        if (!guild || !ticketData || !ticketData.channel || !ticketData.transcriptUrl) {
            console.error('LogSistemi.logTicketTranscript: Geçersiz parametreler');
            return;
        }

        const { channel: ticketChannel, transcriptUrl, viewUrl, closedBy, ticketOwner, messageCount, attachmentCount } = ticketData;

        const embed = new EmbedBuilder()
            .setTitle('📄 Ticket Transcript Oluşturuldu')
            .setDescription(`**Ticket Sahibi:** ${ticketOwner ? `${ticketOwner.user.tag} (${ticketOwner.user.id})` : 'Bilinmiyor'}\n**Kanal:** #${ticketChannel.name}`)
            .addFields(
                { name: '🔗 Transcript Linki', value: `[Konuşma Geçmişini Görüntüle](${viewUrl || transcriptUrl})`, inline: false },
                { name: '💬 Mesaj Sayısı', value: messageCount?.toString() || '0', inline: true },
                { name: '📎 Dosya Sayısı', value: attachmentCount?.toString() || '0', inline: true },
                { name: '🆔 Kanal ID', value: ticketChannel.id, inline: true },
                { name: '🕐 Oluşturulma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: false }
            )
            .setThumbnail(ticketOwner ? ticketOwner.user.displayAvatarURL({ dynamic: true }) : closedBy.displayAvatarURL({ dynamic: true }))
            .setColor('#0099ff')
            .setTimestamp();

        await this.sendLog('ticketTranscriptLog', embed);
    }

    async logTicketClaim(guild, claimData) {
        if (!guild || !claimData || !claimData.staff || !claimData.channel) {
            console.error('LogSistemi.logTicketClaim: Geçersiz parametreler');
            return;
        }

        const { staff, channel, timestamp } = claimData;

        const embed = new EmbedBuilder()
            .setTitle('🎯 Ticket Claim Edildi')
            .setDescription(`**Yetkili:** ${staff.user.tag} (${staff.user.id})\n**Mention:** <@${staff.user.id}>`)
            .addFields(
                { name: '📂 Ticket Kanalı', value: `<#${channel.id}>`, inline: true },
                { name: '🆔 Kanal ID', value: channel.id, inline: true },
                { name: '🕐 Claim Zamanı', value: `<t:${Math.floor(timestamp / 1000)}:F>`, inline: false },
                { name: '⚡ İşlem Durumu', value: '✅ Başarılı - Yetkili kanala erişim kazandı', inline: false }
            )
            .setThumbnail(staff.user.displayAvatarURL({ dynamic: true }))
            .setColor('#00ff00')
            .setTimestamp();

        await this.sendLog('claimLog', embed);
    }

    async logTicketDurt(guild, durtData) {
        if (!guild || !durtData || !durtData.staff || !durtData.ticketOwner || !durtData.channel) {
            console.error('LogSistemi.logTicketDurt: Geçersiz parametreler');
            return;
        }

        const { staff, ticketOwner, channel, type, timestamp } = durtData;

        let title, description, color;
        
        if (type === 'staff_to_owner') {
            title = '🔔 Yetkili → Ticket Sahibi Dürtme';
            description = `**Dürtme Yapan Yetkili:** ${staff.user.tag} (${staff.user.id})\n**Dürtülen Kullanıcı:** ${ticketOwner.user.tag} (${ticketOwner.user.id})`;
            color = '#ff9900';
        } else if (type === 'owner_to_staff') {
            title = '🔔 Ticket Sahibi → Yetkili Dürtme';
            description = `**Dürtme Yapan:** ${ticketOwner.user.tag} (${ticketOwner.user.id})\n**Dürtülen Yetkili:** ${staff.user.tag} (${staff.user.id})`;
            color = '#0099ff';
        }

        const embed = new EmbedBuilder()
            .setTitle(title)
            .setDescription(description)
            .addFields(
                { name: '📂 Ticket Kanalı', value: `<#${channel.id}>`, inline: true },
                { name: '🆔 Kanal ID', value: channel.id, inline: true },
                { name: '🕐 Dürtme Zamanı', value: `<t:${Math.floor(timestamp / 1000)}:F>`, inline: false },
                { name: '📨 Bildirim Durumu', value: '✅ DM bildirimi gönderildi', inline: true },
                { name: '📋 Dürtme Türü', value: type === 'staff_to_owner' ? 'Yetkili → Kullanıcı' : 'Kullanıcı → Yetkili', inline: true }
            )
            .setThumbnail(type === 'staff_to_owner' ? staff.user.displayAvatarURL({ dynamic: true }) : ticketOwner.user.displayAvatarURL({ dynamic: true }))
            .setColor(color)
            .setTimestamp();

        await this.sendLog('durtLog', embed);
    }

    async logBanAffi(guild, banAffiData) {
        const { executor, unbannedUsers, totalBans, successCount, failCount, timestamp } = banAffiData;

        const mainEmbed = new EmbedBuilder()
            .setTitle('🔓 Toplu Ban Affı İşlemi')
            .setDescription(`**${executor.user.tag}** tarafından toplu ban affı işlemi gerçekleştirildi.`)
            .addFields(
                { name: '👤 İşlemi Yapan', value: `${executor.user.tag} (${executor.user.id})`, inline: true },
                { name: '📊 Toplam Ban Sayısı', value: totalBans.toString(), inline: true },
                { name: '✅ Başarılı Afflar', value: successCount.toString(), inline: true },
                { name: '❌ Başarısız Afflar', value: failCount.toString(), inline: true },
                { name: '📈 Başarı Oranı', value: `${Math.round((successCount / totalBans) * 100)}%`, inline: true },
                { name: '🕐 İşlem Zamanı', value: `<t:${Math.floor(timestamp / 1000)}:F>`, inline: true }
            )
            .setThumbnail(executor.user.displayAvatarURL({ dynamic: true }))
            .setColor(successCount === totalBans ? '#00ff00' : failCount > 0 ? '#ffaa00' : '#ff0000')
            .setTimestamp();

        await this.sendLog('banAffiLog', mainEmbed);

        const detailedLogs = unbannedUsers.slice(0, 10);
        for (const unbanData of detailedLogs) {
            const detailEmbed = new EmbedBuilder()
                .setTitle('🔓 Ban Affı Detayı')
                .setDescription(`**${unbanData.user.tag}** kullanıcısının banı affedildi.`)
                .addFields(
                    { name: '👤 Affedilen Kullanıcı', value: `${unbanData.user.tag} (${unbanData.user.id})`, inline: true },
                    { name: '👮 İşlemi Yapan', value: `${executor.user.tag}`, inline: true },
                    { name: '📅 Orijinal Ban Tarihi', value: unbanData.banDate ? `<t:${Math.floor(new Date(unbanData.banDate).getTime() / 1000)}:F>` : 'Bilinmiyor', inline: true },
                    { name: '📝 Ban Sebebi', value: unbanData.reason || 'Sebep belirtilmemiş', inline: false },
                    { name: '🕐 Affedilme Zamanı', value: `<t:${Math.floor(timestamp / 1000)}:F>`, inline: true },
                    { name: '✅ Durum', value: unbanData.success ? 'Başarılı' : 'Başarısız', inline: true }
                )
                .setThumbnail(unbanData.user.displayAvatarURL({ dynamic: true }))
                .setColor(unbanData.success ? '#00ff00' : '#ff0000')
                .setTimestamp();

            await this.sendLog('banAffiLog', detailEmbed);
            
            await new Promise(resolve => setTimeout(resolve, 100));
        }

        if (unbannedUsers.length > 10) {
            const remainingCount = unbannedUsers.length - 10;
            const summaryEmbed = new EmbedBuilder()
                .setTitle('📋 Ban Affı Özeti')
                .setDescription(`Yukarıdaki detayların yanı sıra **${remainingCount}** kullanıcının daha banı affedildi.`)
                .addFields(
                    { name: '📊 Toplam İşlem', value: unbannedUsers.length.toString(), inline: true },
                    { name: '📋 Gösterilen Detay', value: '10', inline: true },
                    { name: '📝 Kalan', value: remainingCount.toString(), inline: true }
                )
                .setColor('#0099ff')
                .setTimestamp();

            await this.sendLog('banAffiLog', summaryEmbed);
        }
    }

    async logBanAdd(user, executor, reason, deleteMessageDays = 0) {
        const embed = new EmbedBuilder()
            .setTitle('🔨 Kullanıcı Yasaklandı')
            .setDescription(`**${user.tag}** sunucudan yasaklandı.`)
            .addFields(
                { name: '👤 Yasaklanan Kullanıcı', value: `${user.tag} (${user.id})\n<@${user.id}>`, inline: true },
                { name: '👮 Yasaklayan Yetkili', value: `${executor.tag} (${executor.id})\n<@${executor.id}>`, inline: true },
                { name: '📝 Yasaklama Sebebi', value: reason || 'Sebep belirtilmemiş', inline: false },
                { name: '🕐 Yasaklama Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
            )
            .setThumbnail(user.displayAvatarURL({ dynamic: true }))
            .setColor('#8B0000')
            .setTimestamp();

        if (deleteMessageDays > 0) {
            embed.addFields({
                name: '🗑️ Silinen Mesajlar',
                value: `Son ${deleteMessageDays} günlük mesajlar silindi`,
                inline: true
            });
        }

        await this.sendLog('banLog', embed);
    }

    async logKickCommand(user, executor, reason) {
        const embed = new EmbedBuilder()
            .setTitle('👢 Kullanıcı Atıldı')
            .setDescription(`**${user.tag}** sunucudan atıldı.`)
            .addFields(
                { name: '👤 Atılan Kullanıcı', value: `${user.tag} (${user.id})\n<@${user.id}>`, inline: true },
                { name: '👮 Atan Yetkili', value: `${executor.tag} (${executor.id})\n<@${executor.id}>`, inline: true },
                { name: '📝 Atılma Sebebi', value: reason || 'Sebep belirtilmemiş', inline: false },
                { name: '🕐 Atılma Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true },
                { name: '🔗 Not', value: 'Kullanıcı davet linki ile tekrar sunucuya katılabilir.', inline: false }
            )
            .setThumbnail(user.displayAvatarURL({ dynamic: true }))
            .setColor('#FF4500')
            .setTimestamp();

        await this.sendLog('kickLog', embed);
    }

    async logUnbanOperation(targetUser, action, reason, executor, deleteMessageDays = null) {
        try {
            let embed;
            
            if (action === 'ban') {
                return;
            } else if (action === 'unban') {
                embed = new EmbedBuilder()
                    .setTitle('🔓 Kullanıcının Banı Açıldı')
                    .setDescription(`**${targetUser.tag}** kullanıcısının banı açıldı.`)
                    .addFields(
                        { name: '👤 Ban Açılan Kullanıcı', value: `${targetUser.tag} (${targetUser.id})\n<@${targetUser.id}>`, inline: true },
                        { name: '👮 Ban Açan Yetkili', value: `${executor.tag} (${executor.id})\n<@${executor.id}>`, inline: true },
                        { name: '📝 Unban Sebebi', value: reason || 'Sebep belirtilmemiş', inline: false },
                        { name: '🕐 Unban Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true },
                        { name: '✅ Durum', value: 'Kullanıcı artık sunucuya katılabilir', inline: false }
                    )
                    .setThumbnail(targetUser.displayAvatarURL({ dynamic: true }))
                    .setColor('#00FF00')
                    .setTimestamp();
            } else if (action === 'info') {
                return;
            }

            await this.sendLog('unbanLog', embed);
        } catch (error) {
            console.error('Unban log hatası:', error);
        }
    }

    async logRoleManagementOperation(data) {
        try {
            const { type, role, executor, affectedUsers, guild, successCount, failCount, errors } = data;
            
            let logType, title, color, emoji;
            
            switch (type) {
                case 'ROLE_ADD_ALL':
                    logType = 'herkeseRolVerLog';
                    title = '👥 Herkese Rol Verme İşlemi';
                    color = '#00ff00';
                    emoji = '✅';
                    break;
                case 'ROLE_REMOVE_ALL':
                    logType = 'herkestenRolAlLog';
                    title = '👤 Herkesten Rol Alma İşlemi';
                    color = '#ff6600';
                    emoji = '🗑️';
                    break;
                default:
                    return;
            }

            const embed = new EmbedBuilder()
                .setTitle(title)
                .setColor(color)
                .addFields(
                    { name: '👮 Yetkili', value: `${executor.tag} (${executor.id})`, inline: true },
                    { name: '🎭 Rol', value: `${role.name} (${role.id})`, inline: true },
                    { name: '🏷️ Rol Rengi', value: role.hexColor || '#000000', inline: true },
                    { name: '👥 Toplam Hedef', value: affectedUsers.toString(), inline: true },
                    { name: '✅ Başarılı', value: successCount.toString(), inline: true },
                    { name: '❌ Başarısız', value: failCount.toString(), inline: true }
                )
                .setTimestamp()
                .setFooter({ text: `RoveBot\'s • Rol Yönetimi`, iconURL: guild.iconURL() });

            if (errors && errors.length > 0) {
                const errorText = errors.slice(0, 5).map(err => `• ${err}`).join('\n');
                embed.addFields({ 
                    name: '⚠️ Hatalar', 
                    value: errors.length > 5 ? `${errorText}\n... ve ${errors.length - 5} hata daha` : errorText, 
                    inline: false 
                });
            }

            await this.sendLog(logType, embed);
        } catch (error) {
            console.error('Rol yönetimi log hatası:', error);
        }
    }

    async logGiveawayOperation(data) {
        try {
            const { type, giveawayId, title, executor, guild, winners, participantCount, prize, endDate } = data;
            
            let logType = 'cekilisLog';
            let embedTitle, color, emoji;
            
            switch (type) {
                case 'GIVEAWAY_CREATE':
                    embedTitle = '🎉 Çekiliş Oluşturuldu';
                    color = '#00ff00';
                    emoji = '🎁';
                    break;
                case 'GIVEAWAY_END':
                    embedTitle = '🏆 Çekiliş Sonuçlandı';
                    color = '#ff6600';
                    emoji = '🎊';
                    break;
                case 'GIVEAWAY_CANCEL':
                    embedTitle = '❌ Çekiliş İptal Edildi';
                    color = '#ff0000';
                    emoji = '🚫';
                    break;
                default:
                    return;
            }

            const embed = new EmbedBuilder()
                .setTitle(embedTitle)
                .setColor(color)
                .addFields(
                    { name: '🎯 Çekiliş ID', value: giveawayId, inline: true },
                    { name: '📝 Başlık', value: title.length > 100 ? title.substring(0, 97) + '...' : title, inline: true },
                    { name: '👮 Yetkili', value: `${executor.tag} (${executor.id})`, inline: true }
                )
                .setTimestamp()
                .setFooter({ text: `RoveBot\'s • Çekiliş Sistemi`, iconURL: guild.iconURL() });

            if (type === 'GIVEAWAY_CREATE') {
                embed.addFields(
                    { name: '🎁 Ödül', value: prize || 'Belirtilmemiş', inline: true },
                    { name: '⏰ Bitiş Tarihi', value: endDate ? `<t:${Math.floor(new Date(endDate).getTime() / 1000)}:F>` : 'Belirtilmemiş', inline: true },
                    { name: '📊 Durum', value: '🟢 Aktif', inline: true }
                );
            } else if (type === 'GIVEAWAY_END') {
                embed.addFields(
                    { name: '👥 Katılımcı Sayısı', value: participantCount?.toString() || '0', inline: true },
                    { name: '🏆 Kazanan Sayısı', value: winners?.length?.toString() || '0', inline: true },
                    { name: '📊 Durum', value: '🔴 Sonuçlandı', inline: true }
                );

                if (winners && winners.length > 0) {
                    const winnerList = winners.slice(0, 10).map(w => `<@${w}>`).join('\n');
                    embed.addFields({ 
                        name: '🏆 Kazananlar', 
                        value: winners.length > 10 ? `${winnerList}\n... ve ${winners.length - 10} kişi daha` : winnerList, 
                        inline: false 
                    });
                }
            }

            await this.sendLog(logType, embed);
        } catch (error) {
            console.error('Çekiliş log hatası:', error);
        }
    }

    async logTicketMemberAction(guildId, action, targetUser, moderator, channel) {
        try {
            const embed = new EmbedBuilder()
                .setColor(action === 'ekle' ? '#00ff00' : '#ff0000')
                .setTitle(`🎫 Ticket ${action === 'ekle' ? 'Üye Eklendi' : 'Üye Kaldırıldı'}`)
                .addFields(
                    { name: '👤 Kullanıcı', value: `<@${targetUser.id}> (${targetUser.username})`, inline: true },
                    { name: '👮 Yetkili', value: `<@${moderator.id}> (${moderator.username})`, inline: true },
                    { name: '📍 Kanal', value: `<#${channel.id}> (${channel.name})`, inline: true },
                    { name: '⚡ İşlem', value: action === 'ekle' ? 'Kanala erişim verildi' : 'Kanaldan erişim kaldırıldı', inline: false },
                    { name: '🔐 İzinler', value: action === 'ekle' ? 
                        '• Kanalı görme ✅\n• Mesaj gönderme ✅\n• Mesaj geçmişini okuma ✅' : 
                        '• Tüm izinler kaldırıldı ❌', inline: false }
                )
                .setTimestamp()
                .setFooter({ text: `Ticket ID: ${channel.id}` });

            await this.sendLog('ticketUyeLog', embed);
        } catch (error) {
            console.error('Ticket üye işlemi log hatası:', error);
        }
    }

    async logRoleOperation(guildId, action, targetUser, moderator, roles, channel) {
        try {
            const roleList = roles.map(role => `<@&${role.id}>`).join(', ');
            const embed = new EmbedBuilder()
                .setColor(action === 'ver' ? '#00ff00' : '#ff0000')
                .setTitle(`🎭 Rol ${action === 'ver' ? 'Verildi' : 'Alındı'}`)
                .addFields(
                    { name: '👤 Kullanıcı', value: `<@${targetUser.id}> (${targetUser.username})`, inline: true },
                    { name: '👮 Yetkili', value: `<@${moderator.id}> (${moderator.username})`, inline: true },
                    { name: '📍 Kanal', value: `<#${channel.id}>`, inline: true },
                    { name: '🎭 Rol(ler)', value: roleList, inline: false },
                    { name: '⚡ İşlem', value: action === 'ver' ? 'Roller kullanıcıya verildi' : 'Roller kullanıcıdan alındı', inline: false }
                )
                .setTimestamp()
                .setFooter({ text: `Kullanıcı ID: ${targetUser.id}` });

            await this.sendLog('rolIslemLog', embed);
        } catch (error) {
            console.error('Rol işlemi log hatası:', error);
        }
    }

    async logMessageDeleteOperation(guildId, moderator, channel, deletedCount, targetUser = null) {
        try {
            const embed = new EmbedBuilder()
                .setColor('#ff0000')
                .setTitle('🗑️ Mesaj Silme İşlemi')
                .addFields(
                    { name: '👮 Yetkili', value: `<@${moderator.id}> (${moderator.username})`, inline: true },
                    { name: '📍 Kanal', value: `<#${channel.id}>`, inline: true },
                    { name: '🔢 Silinen Mesaj', value: `${deletedCount} mesaj`, inline: true },
                    { name: '👤 Hedef Kullanıcı', value: targetUser ? `<@${targetUser.id}> (${targetUser.username})` : 'Tüm kullanıcılar', inline: false },
                    { name: '⚡ İşlem Türü', value: targetUser ? 'Kullanıcı mesajları silindi' : 'Kanal mesajları silindi', inline: false }
                )
                .setTimestamp()
                .setFooter({ text: `Kanal ID: ${channel.id}` });

            await this.sendLog('mesajSilLog', embed);
        } catch (error) {
            console.error('Mesaj silme işlemi log hatası:', error);
        }
    }

    async logVoiceOperation(guildId, moderator, action, voiceChannel, affectedUsers, targetChannel = null) {
        try {
            let actionText = '';
            let color = '#ff0000';
            
            switch (action) {
                case 'disconnect':
                    actionText = 'Bağlantı kesildi';
                    color = '#ff0000';
                    break;
                case 'mute':
                    actionText = 'Susturuldu';
                    color = '#ffaa00';
                    break;
                case 'unmute':
                    actionText = 'Susturma kaldırıldı';
                    color = '#00ff00';
                    break;
                case 'move':
                    actionText = 'Taşındı';
                    color = '#0099ff';
                    break;
            }

            const embed = new EmbedBuilder()
                .setColor(color)
                .setTitle(`🔊 Ses Kanalı ${actionText}`)
                .addFields(
                    { name: '👮 Yetkili', value: `<@${moderator.id}> (${moderator.username})`, inline: true },
                    { name: '🔊 Kaynak Kanal', value: `<#${voiceChannel.id}>`, inline: true },
                    { name: '👥 Etkilenen Kullanıcı', value: `${affectedUsers} kişi`, inline: true },
                    { name: '⚡ İşlem', value: actionText, inline: false }
                )
                .setTimestamp()
                .setFooter({ text: `Kanal ID: ${voiceChannel.id}` });

            if (targetChannel) {
                embed.addFields({ name: '🎯 Hedef Kanal', value: `<#${targetChannel.id}>`, inline: false });
            }

            await this.sendLog('sesIslemLog', embed);
        } catch (error) {
            console.error('Ses işlemi log hatası:', error);
        }
    }

    async logAutoRole(member, role) {
        try {
            const embed = new EmbedBuilder()
                .setTitle('🤖 Otomatik Rol Verildi')
                .setDescription(`**Kullanıcı:** ${member.user.tag} (${member.user.id})\n**Mention:** <@${member.user.id}>`)
                .addFields(
                    { name: '🎭 Verilen Rol', value: `<@&${role.id}> (${role.name})`, inline: true },
                    { name: '🕐 Verilme Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true },
                    { name: '🔄 İşlem Türü', value: 'Otomatik Rol Sistemi', inline: true }
                )
                .setThumbnail(member.user.displayAvatarURL({ dynamic: true }))
                .setColor('#00ff00')
                .setTimestamp()
                .setFooter({ text: `Kullanıcı ID: ${member.user.id}` });

            await this.sendLog('otorolLog', embed);
        } catch (error) {
            console.error('Otorol log hatası:', error);
        }
    }

    async logNuke(guildId, moderator, oldChannel, newChannel) {
        try {
            const embed = new EmbedBuilder()
                .setColor('#ff0000')
                .setTitle('💥 Kanal Nuke İşlemi')
                .setDescription('Bir kanal nuke edildi ve yeniden oluşturuldu.')
                .addFields(
                    { name: '👤 Moderatör', value: `<@${moderator.id}> (${moderator.tag})`, inline: true },
                    { name: '📝 Eski Kanal Adı', value: oldChannel.name, inline: true },
                    { name: '🆔 Eski Kanal ID', value: oldChannel.id, inline: true },
                    { name: '📝 Yeni Kanal Adı', value: newChannel.name, inline: true },
                    { name: '🆔 Yeni Kanal ID', value: `<#${newChannel.id}>`, inline: true },
                    { name: '📂 Kategori', value: newChannel.parent ? `<#${newChannel.parent.id}>` : 'Yok', inline: true }
                )
                .setTimestamp()
                .setFooter({ text: `Sunucu ID: ${guildId}` });

            if (oldChannel.topic) {
                embed.addFields({ name: '📄 Kanal Açıklaması', value: oldChannel.topic.substring(0, 1024), inline: false });
            }

            await this.sendLog('nukeLog', embed);
        } catch (error) {
            console.error('Nuke işlemi log hatası:', error);
        }
    }

    async logSunucuDurumEmbed(guildId, moderator, settings) {
        try {
            const embed = new EmbedBuilder()
                .setColor('#00ff00')
                .setTitle('🖥️ Sunucu Durum Embed Ayarları')
                .setDescription('Sunucu durum embed ayarları güncellendi.')
                .addFields(
                    { name: '👤 Moderatör', value: `<@${moderator.id}> (${moderator.tag})`, inline: true },
                    { name: '🌐 IP Adresi', value: settings.ipAddress || 'GIZLI', inline: true },
                    { name: '🔗 CFX Sunucu Linki', value: settings.cfxLink, inline: true },
                    { name: '🎫 Ticket Oda ID', value: settings.ticketRoomId ? `<#${settings.ticketRoomId}>` : 'Belirtilmedi', inline: true }
                )
                .setTimestamp()
                .setFooter({ text: `Sunucu ID: ${guildId}` });

            await this.sendLog('sunucuDurumEmbedLog', embed);
        } catch (error) {
            console.error('Sunucu durum embed log hatası:', error);
        }
    }

    async logSunucuDurum(guildId, moderator, status, everyoneMention) {
        try {
            const statusEmojis = {
                'Aktif': '🟢',
                'Bakım': '🔴',
                'Restart': '🟡'
            };

            const embed = new EmbedBuilder()
                .setColor(status === 'Aktif' ? '#00ff00' : status === 'Bakım' ? '#ff0000' : '#ffff00')
                .setTitle('📊 Sunucu Durum Güncellendi')
                .setDescription('Sunucu durumu değiştirildi.')
                .addFields(
                    { name: '👤 Moderatör', value: `<@${moderator.id}> (${moderator.tag})`, inline: true },
                    { name: '📊 Yeni Durum', value: `${statusEmojis[status]} ${status}`, inline: true },
                    { name: '📢 Everyone Mention', value: everyoneMention ? 'Evet' : 'Hayır', inline: true },
                    { name: '🕐 Güncelleme Zamanı', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: false }
                )
                .setTimestamp()
                .setFooter({ text: `Sunucu ID: ${guildId}` });

            await this.sendLog('sunucuDurumLog', embed);
        } catch (error) {
            console.error('Sunucu durum log hatası:', error);
        }
    }

    isActive() {
        return this.config && this.config.LogSistemi.aktif;
    }

    reloadConfig() {
        this.config = this.loadConfig();
    }
}

module.exports = LogSistemi;
