const fs = require('fs');
const path = require('path');
const { EmbedBuilder } = require('discord.js');

class KufurSistemi {
    constructor(client) {
        this.client = client;
        this.config = require('../config.json');
        this.kullaniciVerileri = new Map();
        
        this.exceptionUsers = new Set();
        this.exceptionChannels = new Set();
        this.exceptionRoles = new Set();
        
        this.normalKufurler = this.loadJsonFile('./src/kufurlistesi.json');
        this.diniKufurler = this.loadJsonFile('./src/dini_kufurler.json');
        this.milliKufurler = this.loadJsonFile('./src/milli_kufurler.json');
        
        console.log('🛡️ Küfür Sistemi başlatıldı');
        console.log(`📝 Normal küfürler: ${this.normalKufurler.length}`);
        console.log(`🕌 Dini küfürler: ${this.diniKufurler.length}`);
        console.log(`🇹🇷 Milli küfürler: ${this.milliKufurler.length}`);
    }
    
    loadJsonFile(filePath) {
        try {
            const fullPath = path.resolve(filePath);
            const data = fs.readFileSync(fullPath, 'utf8');
            return JSON.parse(data);
        } catch (error) {
            console.error(`❌ JSON dosyası yüklenemedi: ${filePath}`, error);
            return [];
        }
    }
    
    isActive() {
        return this.config.KufurSistemi.aktif;
    }
    
    toggleSystem(aktif) {
        this.config.KufurSistemi.aktif = aktif;
        this.saveConfig();
        return aktif;
    }
    
    saveConfig() {
        try {
            fs.writeFileSync('./config.json', JSON.stringify(this.config, null, 4));
        } catch (error) {
            console.error('❌ Config dosyası kaydedilemedi:', error);
        }
    }
    
    getUserData(userId) {
        if (!this.kullaniciVerileri.has(userId)) {
            this.kullaniciVerileri.set(userId, {
                kufurSayisi: 0,
                sonKufur: null,
                timeoutBitisTarihi: null
            });
        }
        return this.kullaniciVerileri.get(userId);
    }
    
    checkMessage(message) {
        if (!this.isActive()) return null;
        
        if (this.exceptionUsers.has(message.author.id)) {
            return null;
        }
        
        if (this.exceptionChannels.has(message.channel.id)) {
            return null;
        }
        
        if (message.member && message.member.roles.cache.some(role => this.exceptionRoles.has(role.id))) {
            return null;
        }
        
        const content = message.content.toLowerCase();
        const words = content.split(/\s+/);
        
        for (const word of words) {
            for (const kufur of this.diniKufurler) {
                if (word.includes(kufur.toLowerCase())) {
                    return {
                        type: 'dini',
                        word: kufur,
                        foundWord: word
                    };
                }
            }
        }
        
        for (const word of words) {
            for (const kufur of this.milliKufurler) {
                if (word.includes(kufur.toLowerCase())) {
                    return {
                        type: 'milli',
                        word: kufur,
                        foundWord: word
                    };
                }
            }
        }
        
        for (const word of words) {
            for (const kufur of this.normalKufurler) {
                if (word.includes(kufur.toLowerCase())) {
                    return {
                        type: 'normal',
                        word: kufur,
                        foundWord: word
                    };
                }
            }
        }
        
        return null;
    }
    
    calculateTimeout(userId, kufurType) {
        const userData = this.getUserData(userId);
        const timeouts = this.config.KufurSistemi.timeoutSureleri;
        
        if (kufurType === 'dini' || kufurType === 'milli') {
            return timeouts.diniMilli;
        }
        
        userData.kufurSayisi++;
        
        switch (userData.kufurSayisi) {
            case 1:
                return timeouts.birinci;
            case 2:
                return timeouts.ikinci;
            case 3:
                return timeouts.ucuncu;
            case 4:
                return timeouts.dorduncu;
            case 5:
            default:
                return timeouts.besinci;
        }
    }
    
    async applyTimeout(member, duration, reason) {
        if (duration === 0) return false;
        
        try {
            await member.timeout(duration, reason);
            
            const userData = this.getUserData(member.id);
            userData.timeoutBitisTarihi = new Date(Date.now() + duration);
            
            return true;
        } catch (error) {
            console.error('❌ Timeout uygulanamadı:', error);
            return false;
        }
    }
    
    async sendLog(guild, user, kufurData, timeoutDuration, timeoutApplied) {
        try {
            if (this.client.logSistemi && this.client.logSistemi.isActive()) {
                const member = guild.members.cache.get(user.id);
                if (member) {
                    await this.client.logSistemi.logProfanity(
                        member, 
                        kufurData.foundWord, 
                        kufurData.type.charAt(0).toUpperCase() + kufurData.type.slice(1), 
                        timeoutDuration
                    );
                }
                return;
            }

            const logChannelId = this.config.KufurSistemi.kufurlogchannelid;
            if (!logChannelId || logChannelId === 'KANAL_ID_BURAYA') return;
            
            const logChannel = guild.channels.cache.get(logChannelId);
            if (!logChannel) return;
            
            const userData = this.getUserData(user.id);
            
            let color = 0xffff00;
            if (kufurData.type === 'dini') color = 0xff0000;
            else if (kufurData.type === 'milli') color = 0xff4500;
            else if (timeoutApplied) color = 0xff6600;
            
            let categoryEmoji = '⚠️';
            if (kufurData.type === 'dini') categoryEmoji = '🕌';
            else if (kufurData.type === 'milli') categoryEmoji = '🇹🇷';
            
            let timeoutText = 'Uyarı';
            if (timeoutDuration > 0) {
                const minutes = Math.floor(timeoutDuration / 60000);
                timeoutText = `${minutes} dakika timeout`;
            }
            
            const embed = new EmbedBuilder()
                .setColor(color)
                .setTitle(`${categoryEmoji} Küfür Tespiti`)
                .setDescription(`**${user.tag}** tarafından küfür kullanıldı`)
                .addFields(
                    {
                        name: '👤 Kullanıcı',
                        value: `${user} (${user.id})`,
                        inline: true
                    },
                    {
                        name: '🔍 Tespit Edilen Kelime',
                        value: `\`${kufurData.foundWord}\``,
                        inline: true
                    },
                    {
                        name: '📂 Kategori',
                        value: kufurData.type.charAt(0).toUpperCase() + kufurData.type.slice(1),
                        inline: true
                    },
                    {
                        name: '⏱️ Uygulanan Ceza',
                        value: timeoutText,
                        inline: true
                    },
                    {
                        name: '📊 Toplam Küfür',
                        value: `${userData.kufurSayisi}`,
                        inline: true
                    },
                    {
                        name: '📅 Tarih',
                        value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                        inline: true
                    }
                )
                .setThumbnail(user.displayAvatarURL())
                .setFooter({ 
                    text: 'RoveBot\'s Küfür Sistemi',
                    iconURL: guild.iconURL() 
                })
                .setTimestamp();
            
            await logChannel.send({ embeds: [embed] });
        } catch (error) {
            console.error('❌ Log gönderilemedi:', error);
        }
    }
    
    async processMessage(message) {
        if (!this.isActive()) return false;
        if (message.author.bot) return false;
        if (!message.guild) return false;
        
        const kufurData = this.checkMessage(message);
        if (!kufurData) return false;
        
        try {
            await message.delete();
            
            const member = message.member;
            const userData = this.getUserData(message.author.id);
            
            const timeoutDuration = this.calculateTimeout(message.author.id, kufurData.type);
            
            let timeoutApplied = false;
            if (timeoutDuration > 0) {
                const reason = `Küfür kullanımı: ${kufurData.type} kategori - ${kufurData.word}`;
                timeoutApplied = await this.applyTimeout(member, timeoutDuration, reason);
            }
            
            userData.sonKufur = {
                kelime: kufurData.word,
                kategori: kufurData.type,
                tarih: new Date()
            };
            
            await this.sendLog(message.guild, message.author, kufurData, timeoutDuration, timeoutApplied);
            
            try {
                let dmMessage = `⚠️ **${message.guild.name}** sunucusunda küfür kullandığınız tespit edildi.\n`;
                dmMessage += `📝 **Kategori:** ${kufurData.type.charAt(0).toUpperCase() + kufurData.type.slice(1)}\n`;
                
                if (timeoutApplied) {
                    const minutes = Math.floor(timeoutDuration / 60000);
                    dmMessage += `⏱️ **Timeout:** ${minutes} dakika\n`;
                } else {
                    dmMessage += `⚠️ **Durum:** Uyarı\n`;
                }
                
                dmMessage += `\n🔄 **Toplam Küfür Sayınız:** ${userData.kufurSayisi}`;
                dmMessage += `\n\n📋 Lütfen sunucu kurallarına uygun davranın.`;
                
                await message.author.send(dmMessage);
            } catch (dmError) {
            }
            
            return true;
        } catch (error) {
            console.error('❌ Küfür işlemi sırasında hata:', error);
            return false;
        }
    }
    
    getUserStats(userId) {
        const userData = this.getUserData(userId);
        return {
            kufurSayisi: userData.kufurSayisi,
            sonKufur: userData.sonKufur,
            timeoutBitisTarihi: userData.timeoutBitisTarihi
        };
    }
    
    getSystemStats() {
        let toplamKufur = 0;
        let aktifTimeoutlar = 0;
        const now = new Date();
        
        for (const [userId, userData] of this.kullaniciVerileri) {
            toplamKufur += userData.kufurSayisi;
            if (userData.timeoutBitisTarihi && userData.timeoutBitisTarihi > now) {
                aktifTimeoutlar++;
            }
        }
        
        return {
            toplamKullanici: this.kullaniciVerileri.size,
            toplamKufur,
            aktifTimeoutlar,
            sistemAktif: this.isActive()
        };
    }
    
    addExceptionUser(userId) {
        this.exceptionUsers.add(userId);
        return true;
    }
    
    removeExceptionUser(userId) {
        return this.exceptionUsers.delete(userId);
    }
    
    getExceptionUsers() {
        return Array.from(this.exceptionUsers);
    }
    
    addExceptionChannel(channelId) {
        this.exceptionChannels.add(channelId);
        return true;
    }
    
    removeExceptionChannel(channelId) {
        return this.exceptionChannels.delete(channelId);
    }
    
    getExceptionChannels() {
        return Array.from(this.exceptionChannels);
    }
    
    addExceptionRole(roleId) {
        this.exceptionRoles.add(roleId);
        return true;
    }
    
    removeExceptionRole(roleId) {
        return this.exceptionRoles.delete(roleId);
    }
    
    getExceptionRoles() {
        return Array.from(this.exceptionRoles);
    }
}

module.exports = KufurSistemi;