const { Collection } = require('discord.js');

class InviteTracker {
    constructor(client) {
        this.client = client;
        this.invites = new Collection();
        this.vanityInvites = new Collection();
    }

    async loadInvites(guild) {
        try {
            const invites = await guild.invites.fetch();
            const inviteMap = new Map();
            
            invites.forEach(invite => {
                inviteMap.set(invite.code, {
                    uses: invite.uses,
                    inviter: invite.inviter,
                    channel: invite.channel,
                    maxUses: invite.maxUses,
                    maxAge: invite.maxAge,
                    createdAt: invite.createdAt
                });
            });

            this.invites.set(guild.id, inviteMap);

            if (guild.vanityURLCode) {
                try {
                    const vanityInvite = await guild.fetchVanityData();
                    this.vanityInvites.set(guild.id, vanityInvite.uses);
                } catch (error) {
                    console.error('Vanity URL alınırken hata:', error);
                }
            }

            console.log(`${guild.name} sunucusu için ${invites.size} davet yüklendi`);
        } catch (error) {
            console.error(`Davetler yüklenirken hata (${guild.name}):`, error);
        }
    }

    async loadAllInvites() {
        for (const guild of this.client.guilds.cache.values()) {
            await this.loadInvites(guild);
        }
    }

    async handleMemberJoin(member) {
        try {
            const guild = member.guild;
            const oldInvites = this.invites.get(guild.id);
            
            if (!oldInvites) {
                console.log(`${guild.name} için eski davet verisi bulunamadı`);
                await this.loadInvites(guild);
                return;
            }

            const newInvites = await guild.invites.fetch();
            let usedInvite = null;

            for (const [code, newInvite] of newInvites) {
                const oldInvite = oldInvites.get(code);
                
                if (oldInvite && newInvite.uses > oldInvite.uses) {
                    usedInvite = {
                        code: code,
                        inviter: newInvite.inviter,
                        channel: newInvite.channel,
                        uses: newInvite.uses,
                        maxUses: newInvite.maxUses,
                        oldUses: oldInvite.uses
                    };
                    break;
                }
            }

            if (!usedInvite && guild.vanityURLCode) {
                try {
                    const vanityData = await guild.fetchVanityData();
                    const oldVanityUses = this.vanityInvites.get(guild.id) || 0;
                    
                    if (vanityData.uses > oldVanityUses) {
                        usedInvite = {
                            code: guild.vanityURLCode,
                            inviter: null,
                            channel: null,
                            uses: vanityData.uses,
                            maxUses: 0,
                            oldUses: oldVanityUses,
                            isVanity: true
                        };
                        this.vanityInvites.set(guild.id, vanityData.uses);
                    }
                } catch (error) {
                    console.error('Vanity URL kontrolü hatası:', error);
                }
            }

            if (!usedInvite) {
                usedInvite = {
                    code: 'Bilinmiyor',
                    inviter: null,
                    channel: null,
                    uses: 0,
                    maxUses: 0,
                    oldUses: 0,
                    isUnknown: true
                };
            }

            if (this.client.logSistemi && this.client.logSistemi.isActive()) {
                await this.client.logSistemi.logInviteUsed(usedInvite, member);
            }

            await this.loadInvites(guild);

        } catch (error) {
            console.error('Davet takibi hatası:', error);
        }
    }

    async handleInviteCreate(invite) {
        try {
            const guild = invite.guild;
            const guildInvites = this.invites.get(guild.id) || new Map();
            
            guildInvites.set(invite.code, {
                uses: invite.uses,
                inviter: invite.inviter,
                channel: invite.channel,
                maxUses: invite.maxUses,
                maxAge: invite.maxAge,
                createdAt: invite.createdAt
            });

            this.invites.set(guild.id, guildInvites);
        } catch (error) {
            console.error('Davet oluşturma takibi hatası:', error);
        }
    }

    async handleInviteDelete(invite) {
        try {
            const guild = invite.guild;
            const guildInvites = this.invites.get(guild.id);
            
            if (guildInvites) {
                guildInvites.delete(invite.code);
            }
        } catch (error) {
            console.error('Davet silme takibi hatası:', error);
        }
    }

    getInviteStats(guildId, userId) {
        const guildInvites = this.invites.get(guildId);
        if (!guildInvites) return null;

        let totalInvites = 0;
        let activeInvites = 0;

        for (const [code, invite] of guildInvites) {
            if (invite.inviter && invite.inviter.id === userId) {
                totalInvites += invite.uses;
                activeInvites++;
            }
        }

        return {
            totalInvites,
            activeInvites
        };
    }

    getTopInviters(guildId, limit = 10) {
        const guildInvites = this.invites.get(guildId);
        if (!guildInvites) return [];

        const inviterStats = new Map();

        for (const [code, invite] of guildInvites) {
            if (invite.inviter) {
                const current = inviterStats.get(invite.inviter.id) || {
                    user: invite.inviter,
                    invites: 0,
                    codes: 0
                };
                
                current.invites += invite.uses;
                current.codes++;
                inviterStats.set(invite.inviter.id, current);
            }
        }

        return Array.from(inviterStats.values())
            .sort((a, b) => b.invites - a.invites)
            .slice(0, limit);
    }
}

module.exports = InviteTracker;