const { Events, EmbedBuilder, ActionRowBuilder, StringSelectMenuBuilder, PermissionFlagsBits } = require('discord.js');
const fs = require('fs');
const path = require('path');
const { checkPermission } = require('../commands/yetkiayarla');

module.exports = {
    name: Events.MessageReactionAdd,
    async execute(reaction, user) {
        try {
            // Bot reaksiyonlarını yoksay
            if (user.bot) return;

            // Kısmi mesaj kontrolü
            if (reaction.partial) {
                try {
                    await reaction.fetch();
                } catch (error) {
                    console.error('Reaction fetch hatası:', error);
                    return;
                }
            }

            const message = reaction.message;

            // Mesajın partial olup olmadığını kontrol et
            if (message.partial) {
                try {
                    await message.fetch();
                } catch (error) {
                    console.error('Message fetch hatası:', error);
                    return;
                }
            }

            // IC İsim sistemi kontrolü
            const icisimPath = path.join(__dirname, '../icisimSistemi.json');
            if (!fs.existsSync(icisimPath)) return;

            const icisimData = JSON.parse(fs.readFileSync(icisimPath, 'utf8'));
            const guildData = icisimData[message.guild.id];

            if (!guildData || !guildData.aktif || guildData.kanalId !== message.channel.id) {
                return;
            }

            // Sadece ✅ ve ❌ emojilerini kontrol et
            const emojiName = reaction.emoji.name;
            const emojiString = reaction.emoji.toString();
            const emojiId = reaction.emoji.id;
            
            // Unicode emojiler için kontrol (✅ ve ❌)
            // Discord.js'de Unicode emojiler için name property'si emoji'nin adını döndürür
            const isCheckmark = emojiName === '✅' || emojiString === '✅' || emojiName === 'white_check_mark' || (emojiId === null && emojiString.includes('✅'));
            const isCross = emojiName === '❌' || emojiString === '❌' || emojiName === 'x' || (emojiId === null && emojiString.includes('❌'));
            
            if (!isCheckmark && !isCross) {
                return;
            }

            console.log('[IC İsim] Reaction eklendi:', {
                emojiName,
                emojiString,
                isCheckmark,
                isCross,
                userId: user.id,
                messageId: message.id,
                channelId: message.channel.id
            });

            // Yetki kontrolü
            const member = await message.guild.members.fetch(user.id).catch(() => null);
            if (!member) {
                console.error('[IC İsim] Member fetch edilemedi:', user.id);
                return;
            }
            
            // Mock interaction object for checkPermission
            const mockInteraction = {
                guild: message.guild,
                member: member,
                user: user
            };

            const hasPermission = checkPermission(mockInteraction, 'icisimonyared');
            console.log('[IC İsim] Yetki kontrolü:', {
                userId: user.id,
                username: user.username,
                hasPermission,
                roles: member.roles.cache.map(r => r.id)
            });

            if (!hasPermission) {
                console.log('[IC İsim] Yetkisiz kullanıcı, reaction kaldırılıyor:', user.id);
                // Yetkisiz kullanıcının reaksiyonunu kaldır
                try {
                    await reaction.users.remove(user.id);
                } catch (error) {
                    console.error('[IC İsim] Reaction kaldırma hatası:', error);
                }
                return;
            }

            // Mesaj sahibini al
            const messageAuthor = message.author;
            if (!messageAuthor || messageAuthor.bot) return;

            const messageAuthorMember = await message.guild.members.fetch(messageAuthor.id).catch(() => null);
            if (!messageAuthorMember) return;

            // İsim metnini al
            const icIsim = message.content.trim();

            if (isCheckmark) {
                // Onay işlemi - kullanıcının ismini değiştir
                try {
                    // Bot izin ve rol hiyerarşisi kontrolü
                    const botMember = message.guild.members.me;
                    const canManageNick = botMember.permissions.has(PermissionFlagsBits.ManageNicknames);
                    const hierarchyOk = botMember.roles.highest.comparePositionTo(messageAuthorMember.roles.highest) > 0;

                    if (!canManageNick || !hierarchyOk) {
                        const reasonText = !canManageNick
                            ? 'Botun "Takma Adları Yönet" izni yok.'
                            : 'Botun rolü, hedef kullanıcının en yüksek rolünden düşük.';

                        // Yalnızca yetkilinin göreceği şekilde DM gönder
                        try {
                            await user.send(`❌ IC İsim onaylanamadı. ${reasonText}`);
                        } catch (dmErr) {
                            // DM kapalıysa kanalda geçici bir bilgilendirme gönderip sil
                            try {
                                const tempMsg = await message.channel.send({
                                    content: `❌ ${user}, IC İsim onaylanamadı. ${reasonText} (DM kapalı)`,
                                    allowedMentions: { users: [user.id], repliedUser: false }
                                });
                                setTimeout(() => tempMsg.delete().catch(() => {}), 10000);
                            } catch (_) {}
                        }
                        // Yetkisiz onay denemesinde reaksiyonu kaldır
                        try { await reaction.users.remove(user.id); } catch (_) {}
                        return;
                    }

                    await messageAuthorMember.setNickname(icIsim, 'IC İsim onaylandı');
                    
                    // Log gönder
                    const configPath = path.join(__dirname, '../../config.json');
                    const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));
                    
                    if (config.LogSistemi.aktif && config.LogSistemi.logKanallari.icisimOnayRedLog) {
                        const logChannel = message.guild.channels.cache.get(config.LogSistemi.logKanallari.icisimOnayRedLog);
                        if (logChannel) {
                            const logEmbed = new EmbedBuilder()
                                .setColor(0x00ff00)
                                .setTitle('✅ IC İsim Onaylandı')
                                .setDescription('IC İsim talebi onaylandı')
                                .addFields({
                                    name: '👤 Kullanıcı',
                                    value: `${messageAuthor} (${messageAuthor.id})`,
                                    inline: true
                                })
                                .addFields({
                                    name: '📝 Yeni İsim',
                                    value: icIsim,
                                    inline: true
                                })
                                .addFields({
                                    name: '👮 Onaylayan',
                                    value: `${user} (${user.id})`,
                                    inline: true
                                })
                                .addFields({
                                    name: '💬 Mesaj',
                                    value: `[Mesaja Git](${message.url})`,
                                    inline: false
                                })
                                .setFooter({ text: 'RoveBot\'s • Log Sistemi' })
                                .setTimestamp();

                            await logChannel.send({ embeds: [logEmbed] });
                        }
                    }

                    // Onay mesajı gönder
                    await message.reply({
                        content: `✅ IC İsim talebiniz onaylandı! İsminiz **${icIsim}** olarak değiştirildi.`,
                        allowedMentions: { repliedUser: false }
                    });

                } catch (error) {
                    console.error('IC İsim onay hatası:', error);
                    // Hata durumunda yalnızca yetkiliye bilgi ver
                    try {
                        await user.send('❌ IC İsim onaylanamadı. İsim değiştirilirken bir hata oluştu.');
                    } catch (dmErr) {
                        try {
                            const tempMsg = await message.channel.send({
                                content: `❌ ${user}, IC İsim onaylanamadı. İsim değiştirilirken bir hata oluştu. (DM kapalı)`,
                                allowedMentions: { users: [user.id], repliedUser: false }
                            });
                            setTimeout(() => tempMsg.delete().catch(() => {}), 10000);
                        } catch (_) {}
                    }
                }

            } else if (isCross) {
                // Red işlemi - red sebebi seçimi
                try {
                    // Red sebepleri select menu
                    const selectMenu = new StringSelectMenuBuilder()
                        .setCustomId(`icisim_red_${message.id}`)
                        .setPlaceholder('Red sebebini seçin...')
                        .addOptions(
                            {
                                label: 'Uygunsuz İsim',
                                value: 'uygunsuz_isim',
                                description: 'İsim uygun değil'
                            },
                            {
                                label: 'Ünlü İsmi',
                                value: 'unlu_ismi',
                                description: 'Ünlü ismi kullanılamaz'
                            }
                        );

                    const row = new ActionRowBuilder().addComponents(selectMenu);

                    // Sadece yetkilinin görebileceği mesaj gönder
                    const redEmbed = new EmbedBuilder()
                        .setColor(0xff0000)
                        .setTitle('❌ IC İsim Red Sebebi')
                        .setDescription('Red sebebini seçin:')
                        .setTimestamp();

                    const replyMessage = await message.reply({
                        embeds: [redEmbed],
                        components: [row],
                        allowedMentions: { repliedUser: false }
                    });

                    // Interaction collector için listener ekle (interactionCreate'de handle edilecek)
                    // Burada sadece mesajı gönderiyoruz, interactionCreate'de handle edeceğiz

                } catch (error) {
                    console.error('IC İsim red select menu hatası:', error);
                }
            }

        } catch (error) {
            console.error('MessageReactionAdd hatası:', error);
        }
    },
};
