const { Events, EmbedBuilder, ActionRowBuilder, ButtonBuilder, ButtonStyle, PermissionFlagsBits } = require('discord.js');
const fs = require('fs');
const path = require('path');
const LogSistemi = require('../logSistemi');
const TranscriptSistemi = require('../transcriptSistemi');

module.exports = {
    name: Events.InteractionCreate,
    async execute(interaction) {
        // Interaction timeout kontrolü
        const INTERACTION_TIMEOUT = 2900; // 2.9 saniye (3 saniye limitinden önce)
        const startTime = Date.now();
        
        if (interaction.isChatInputCommand()) {
            const command = interaction.client.commands.get(interaction.commandName);

            if (!command) {
                console.error(`${interaction.commandName} komutu bulunamadı.`);
                return;
            }

            try {
                // Timeout kontrolü
                if (Date.now() - startTime > INTERACTION_TIMEOUT) {
                    console.warn(`⚠️ ${interaction.commandName} komutu timeout riski nedeniyle atlandı.`);
                    return;
                }
                
                await command.execute(interaction);
                console.log(`✅ ${interaction.user.tag} kullanıcısı /${interaction.commandName} komutunu kullandı.`);
            } catch (error) {
                console.error(`❌ ${interaction.commandName} komutunda hata:`, error);
                
                // Timeout hatası kontrolü
                if (error.code === 10062 || error.message?.includes('Unknown interaction')) {
                    console.warn(`⚠️ ${interaction.commandName} komutu timeout hatası aldı.`);
                    return;
                }
                
                const errorMessage = {
                    content: '❌ Bu komutu çalıştırırken bir hata oluştu!',
                    ephemeral: true
                };

                try {
                    if (interaction.replied || interaction.deferred) {
                        await interaction.followUp(errorMessage);
                    } else {
                        await interaction.reply(errorMessage);
                    }
                } catch (replyError) {
                    console.error('Hata mesajı gönderilirken sorun oluştu:', replyError);
                }
            }
        }

        else if (interaction.isStringSelectMenu()) {
            try {
                // Timeout kontrolü
                if (Date.now() - startTime > INTERACTION_TIMEOUT) {
                    console.warn(`⚠️ Select menu interaction timeout riski nedeniyle atlandı.`);
                    return;
                }
                
                if (interaction.customId === 'ticket_category_select') {
                    await handleTicketCategorySelect(interaction);
                } else if (interaction.customId === 'delete_category_select') {
                    await handleDeleteCategorySelect(interaction);
                } else if (interaction.customId === 'giveaway_select') {
                    await handleGiveawaySelect(interaction);
                } else if (interaction.customId.startsWith('icisim_red_')) {
                    await handleIcIsimRed(interaction);
                }
            } catch (error) {
                console.error('Select menu hatası:', error);
                
                // Timeout hatası kontrolü
                if (error.code === 10062 || error.message?.includes('Unknown interaction')) {
                    console.warn(`⚠️ Select menu timeout hatası aldı.`);
                    return;
                }
                
                try {
                    if (!interaction.replied && !interaction.deferred) {
                        await interaction.reply({
                            content: '❌ Bir hata oluştu!',
                            ephemeral: true
                        });
                    } else {
                        await interaction.followUp({
                            content: '❌ Bir hata oluştu!',
                            ephemeral: true
                        });
                    }
                } catch (replyError) {
                    console.error('Hata mesajı gönderilirken sorun oluştu:', replyError);
                }
            }
        }

        else if (interaction.isButton()) {
            try {
                // Timeout kontrolü
                if (Date.now() - startTime > INTERACTION_TIMEOUT) {
                    console.warn(`⚠️ Button interaction timeout riski nedeniyle atlandı.`);
                    return;
                }
                
                if (interaction.customId === 'close_ticket') {
                    await handleCloseTicket(interaction);
                } else if (interaction.customId === 'delete_ticket') {
                    await handleDeleteTicket(interaction);
                } else if (interaction.customId === 'claim_ticket') {
                    await handleClaimTicket(interaction);
                } else if (interaction.customId === 'durt_ticket') {
                    await handleDurtTicket(interaction);
                } else if (interaction.customId === 'download_avatar') {
                    await handleDownloadAvatar(interaction);
                } else if (interaction.customId === 'download_banner') {
                    await handleDownloadBanner(interaction);
                } else if (interaction.customId === 'download_both') {
                    await handleDownloadBoth(interaction);
                } else if (interaction.customId.startsWith('giveaway_join_')) {
                    await handleGiveawayJoin(interaction);
                } else if (interaction.customId.startsWith('rolbilgi_')) {
                    await handleRolBilgiButton(interaction);
                }
            } catch (error) {
                console.error('Button hatası:', error);
                
                // Timeout hatası kontrolü
                if (error.code === 10062 || error.message?.includes('Unknown interaction')) {
                    console.warn(`⚠️ Button timeout hatası aldı.`);
                    return;
                }
                
                try {
                    if (!interaction.replied && !interaction.deferred) {
                        await interaction.reply({
                            content: '❌ Bir hata oluştu!',
                            ephemeral: true
                        });
                    } else {
                        await interaction.followUp({
                            content: '❌ Bir hata oluştu!',
                            ephemeral: true
                        });
                    }
                } catch (replyError) {
                    console.error('Hata mesajı gönderilirken sorun oluştu:', replyError);
                }
            }
        }
    },
};

async function handleTicketCategorySelect(interaction) {
    // Defer the interaction immediately to prevent timeout
    await interaction.deferReply({ ephemeral: true });
    
    const selectedValue = interaction.values[0];
    
    if (selectedValue === 'reset') {
        return interaction.followUp({
            content: '🔄 Kategori seçimi sıfırlandı.',
            ephemeral: true
        });
    }

    const ticketDataPath = path.join(__dirname, '..', 'ticketSistemi.json');
    let ticketData = {};
    
    if (fs.existsSync(ticketDataPath)) {
        ticketData = JSON.parse(fs.readFileSync(ticketDataPath, 'utf8'));
    }

    const guildId = interaction.guild.id;
    const userId = interaction.user.id;

    if (!ticketData[guildId] || !ticketData[guildId].categories) {
        return interaction.followUp({
            content: '❌ Ticket kategorileri bulunamadı!',
            ephemeral: true
        });
    }

    if (ticketData[guildId].activeTickets && ticketData[guildId].activeTickets[userId]) {
        const existingChannelId = ticketData[guildId].activeTickets[userId];
        const existingChannel = interaction.guild.channels.cache.get(existingChannelId);
        
        if (existingChannel) {
            return interaction.followUp({
                content: `❌ Zaten açık bir ticketiniz var: ${existingChannel}`,
                ephemeral: true
            });
        } else {
            delete ticketData[guildId].activeTickets[userId];
            fs.writeFileSync(ticketDataPath, JSON.stringify(ticketData, null, 2));
        }
    }

    const selectedCategory = ticketData[guildId].categories.find(cat => cat.id === selectedValue);
    
    if (!selectedCategory) {
        return interaction.followUp({
            content: '❌ Seçilen kategori bulunamadı!',
            ephemeral: true
        });
    }

    const categoryChannel = interaction.guild.channels.cache.get(selectedCategory.categoryChannelId);
    
    if (!categoryChannel) {
        return interaction.followUp({
            content: '❌ Kategori kanalı bulunamadı!',
            ephemeral: true
        });
    }

    const channelName = `🎫・ᴛɪᴄᴋᴇᴛ-${interaction.user.username}`.toLowerCase();
    
    const staffRoleId = ticketData[guildId].staffRole;
    const staffRole = staffRoleId ? interaction.guild.roles.cache.get(staffRoleId) : null;

    const permissionOverwrites = [
        {
            id: interaction.guild.roles.everyone.id,
            deny: [PermissionFlagsBits.ViewChannel]
        },
        {
            id: interaction.user.id,
            allow: [PermissionFlagsBits.ViewChannel, PermissionFlagsBits.SendMessages, PermissionFlagsBits.ReadMessageHistory]
        }
    ];

    if (staffRole) {
        permissionOverwrites.push({
            id: staffRole.id,
            allow: [PermissionFlagsBits.ViewChannel, PermissionFlagsBits.SendMessages, PermissionFlagsBits.ReadMessageHistory, PermissionFlagsBits.ManageMessages]
        });
    }

    const ticketChannel = await interaction.guild.channels.create({
        name: channelName,
        parent: categoryChannel.id,
        permissionOverwrites: permissionOverwrites
    });

    if (!ticketData[guildId].activeTickets) {
        ticketData[guildId].activeTickets = {};
    }
    ticketData[guildId].activeTickets[userId] = ticketChannel.id;
    fs.writeFileSync(ticketDataPath, JSON.stringify(ticketData, null, 2));

    const ticketEmbed = new EmbedBuilder()
        .setTitle(`🎫 ${selectedCategory.name} Ticket`)
        .setDescription(`Merhaba ${interaction.user}!\n\nTicketiniz başarıyla oluşturuldu. Lütfen sorununuzu detaylı bir şekilde açıklayın. Yetkili ekibimiz en kısa sürede size yardımcı olacaktır.`)
        .addFields(
            { name: 'Kategori', value: `${selectedCategory.emoji} ${selectedCategory.name}`, inline: true },
            { name: 'Oluşturan', value: interaction.user.toString(), inline: true },
            { name: 'Oluşturulma Tarihi', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: true }
        )
        .setColor('#00ff00')
        .setFooter({ text: 'RoveBot\'s • Ticket Sistemi' })
        .setTimestamp();

    const closeButton = new ButtonBuilder()
        .setCustomId('close_ticket')
        .setLabel('Ticketi Kapat')
        .setStyle(ButtonStyle.Danger)
        .setEmoji('🔒');

    const claimButton = new ButtonBuilder()
        .setCustomId('claim_ticket')
        .setLabel('Claim')
        .setStyle(ButtonStyle.Success)
        .setEmoji('🎯');

    const durtButton = new ButtonBuilder()
        .setCustomId('durt_ticket')
        .setLabel('Dürt')
        .setStyle(ButtonStyle.Secondary)
        .setEmoji('🔔');

    const row = new ActionRowBuilder().addComponents(claimButton, durtButton, closeButton);

    await ticketChannel.send({
        content: staffRole ? `${staffRole}` : '',
        embeds: [ticketEmbed],
        components: [row]
    });

    try {
        const logSistemi = new LogSistemi(interaction.client);
        await logSistemi.logTicketOpen(interaction.guild, {
            user: interaction.user,
            channel: ticketChannel,
            category: selectedCategory.name,
            categoryEmoji: selectedCategory.emoji
        });
    } catch (error) {
        console.error('Ticket açılma logu kaydedilemedi:', error);
    }

    await interaction.followUp({
        content: `✅ Ticketiniz oluşturuldu: ${ticketChannel}`,
        ephemeral: true
    });
}

async function handleDeleteCategorySelect(interaction) {
    const selectedCategoryId = interaction.values[0];
    
    const ticketDataPath = path.join(__dirname, '..', 'ticketSistemi.json');
    let ticketData = JSON.parse(fs.readFileSync(ticketDataPath, 'utf8'));
    
    const guildId = interaction.guild.id;
    const categoryIndex = ticketData[guildId].categories.findIndex(cat => cat.id === selectedCategoryId);
    
    if (categoryIndex === -1) {
        return interaction.reply({
            content: '❌ Kategori bulunamadı!',
            ephemeral: true
        });
    }

    const deletedCategory = ticketData[guildId].categories.splice(categoryIndex, 1)[0];
    fs.writeFileSync(ticketDataPath, JSON.stringify(ticketData, null, 2));

    const embed = new EmbedBuilder()
        .setTitle('🗑️ Kategori Silindi')
        .setDescription(`**${deletedCategory.emoji} ${deletedCategory.name}** kategorisi başarıyla silindi!`)
        .setColor('#ff0000')
        .setTimestamp();

    await interaction.reply({ embeds: [embed], ephemeral: true });
}

async function handleCloseTicket(interaction) {
    await interaction.deferReply({ ephemeral: false });

    const ticketDataPath = path.join(__dirname, '..', 'ticketSistemi.json');
    let ticketData = JSON.parse(fs.readFileSync(ticketDataPath, 'utf8'));
    
    const guildId = interaction.guild.id;
    const staffRoleId = ticketData[guildId]?.staffRole;
    
    const hasPermission = interaction.member.permissions.has(PermissionFlagsBits.Administrator) ||
                         (staffRoleId && interaction.member.roles.cache.has(staffRoleId));

    if (!hasPermission) {
        return interaction.editReply({
            content: '❌ Bu işlemi yapmak için yetkiniz yok!'
        });
    }

    const channelId = interaction.channel.id;
    let ticketOwnerId = null;
    let ticketOwner = null;

    for (const [userId, userChannelId] of Object.entries(ticketData[guildId].activeTickets || {})) {
        if (userChannelId === channelId) {
            ticketOwnerId = userId;
            ticketOwner = interaction.guild.members.cache.get(userId);
            break;
        }
    }

    if (ticketOwner) {
        await interaction.channel.permissionOverwrites.edit(ticketOwner, {
            ViewChannel: false
        });
    }

    let transcriptUrl = null;
    try {
        const transcriptSistemi = new TranscriptSistemi();
        const transcriptResult = await transcriptSistemi.generateTranscript(interaction.channel, ticketOwner);
        transcriptUrl = transcriptResult.url;
        
        if (transcriptResult) {
            const logSistemi = new LogSistemi(interaction.client);
            await logSistemi.logTicketTranscript(interaction.guild, {
                channel: interaction.channel,
                transcriptUrl: transcriptResult.url,
                viewUrl: transcriptResult.viewUrl,
                messageCount: transcriptResult.messageCount,
                attachmentCount: transcriptResult.attachmentCount,
                closedBy: interaction.user,
                ticketOwner: ticketOwner
            });
        }
    } catch (error) {
        console.error('Transcript oluşturma hatası:', error);
    }

    try {
        const logSistemi = new LogSistemi(interaction.client);
        await logSistemi.logTicketClose(interaction.guild, {
            channel: interaction.channel,
            closedBy: interaction.user,
            ticketOwner: ticketOwner,
            transcriptUrl: transcriptUrl
        });
    } catch (error) {
        console.error('Ticket kapatma logu kaydedilemedi:', error);
    }

    const embed = new EmbedBuilder()
        .setTitle('🔒 Ticket Kapatıldı')
        .setDescription('Ticket kapatıldı. Silmek ister misiniz?')
        .setColor('#ff9900')
        .setTimestamp();

    const deleteButton = new ButtonBuilder()
        .setCustomId('delete_ticket')
        .setLabel('Evet, Sil')
        .setStyle(ButtonStyle.Danger)
        .setEmoji('🗑️');

    const row = new ActionRowBuilder().addComponents(deleteButton);

    await interaction.editReply({
        embeds: [embed],
        components: [row]
    });
}

async function handleDeleteTicket(interaction) {
    const ticketDataPath = path.join(__dirname, '..', 'ticketSistemi.json');
    let ticketData = JSON.parse(fs.readFileSync(ticketDataPath, 'utf8'));
    
    const guildId = interaction.guild.id;
    const channelId = interaction.channel.id;

    for (const [userId, userChannelId] of Object.entries(ticketData[guildId].activeTickets || {})) {
        if (userChannelId === channelId) {
            delete ticketData[guildId].activeTickets[userId];
            break;
        }
    }

    fs.writeFileSync(ticketDataPath, JSON.stringify(ticketData, null, 2));

    await interaction.reply({
        content: '🗑️ Ticket 5 saniye içinde silinecek...',
        ephemeral: true
    });

    setTimeout(async () => {
        try {
            await interaction.channel.delete();
        } catch (error) {
            console.error('Ticket silme hatası:', error);
        }
    }, 5000);
}

async function handleClaimTicket(interaction) {
    try {
        const ticketDataPath = path.join(__dirname, '..', 'ticketSistemi.json');
        let ticketData = {};
        
        if (fs.existsSync(ticketDataPath)) {
            ticketData = JSON.parse(fs.readFileSync(ticketDataPath, 'utf8'));
        }

        const guildId = interaction.guild.id;
        const staffRoleId = ticketData[guildId]?.staffRole;

        if (!staffRoleId) {
            return interaction.reply({
                content: '❌ Ticket sistemi için staff rolü tanımlanmamış!',
                ephemeral: true
            });
        }

        const member = interaction.member;
        if (!member.roles.cache.has(staffRoleId)) {
            return interaction.reply({
                content: '❌ Bu işlemi yapmak için yeterli yetkiniz yok!',
                ephemeral: true
            });
        }

        const channel = interaction.channel;
        if (!channel.name.startsWith('🎫・ᴛɪᴄᴋᴇᴛ')) {
            return interaction.reply({
                content: '❌ Bu işlem sadece ticket kanallarında yapılabilir!',
                ephemeral: true
            });
        }

        if (!ticketData[guildId].claimedTickets) {
            ticketData[guildId].claimedTickets = {};
        }

        if (ticketData[guildId].claimedTickets[channel.id]) {
            const claimedBy = ticketData[guildId].claimedTickets[channel.id];
            const claimedMember = interaction.guild.members.cache.get(claimedBy);
            return interaction.reply({
                content: `❌ Bu ticket zaten ${claimedMember ? claimedMember.toString() : 'başka bir yetkili'} tarafından claim edilmiş!`,
                ephemeral: true
            });
        }

        await interaction.deferReply();

        await channel.permissionOverwrites.edit(member.id, {
            ViewChannel: true,
            SendMessages: true,
            ReadMessageHistory: true,
            ManageMessages: true
        });

        ticketData[guildId].claimedTickets[channel.id] = member.id;
        fs.writeFileSync(ticketDataPath, JSON.stringify(ticketData, null, 2));

        await channel.send(`Merhaba, Ben ${member}. Size Nasıl Yardımcı Olabilirim?`);

        try {
            const logSistemi = new LogSistemi(interaction.client);
            await logSistemi.logTicketClaim(interaction.guild, {
                staff: member,
                channel: channel,
                timestamp: Date.now()
            });
        } catch (error) {
            console.error('Ticket claim logu kaydedilemedi:', error);
        }

        await interaction.editReply({
            content: `✅ Ticket başarıyla claim edildi! ${member} artık bu ticketi yönetiyor.`
        });

    } catch (error) {
        console.error('Ticket claim hatası:', error);
        await interaction.reply({
            content: '❌ Ticket claim edilirken bir hata oluştu!',
            ephemeral: true
        });
    }
}

async function handleDurtTicket(interaction) {
    try {
        const ticketDataPath = path.join(__dirname, '..', 'ticketSistemi.json');
        let ticketData = {};
        
        if (fs.existsSync(ticketDataPath)) {
            ticketData = JSON.parse(fs.readFileSync(ticketDataPath, 'utf8'));
        }

        const guildId = interaction.guild.id;
        const staffRoleId = ticketData[guildId]?.staffRole;
        const channel = interaction.channel;

        if (!channel.name.startsWith('🎫・ᴛɪᴄᴋᴇᴛ')) {
            return interaction.reply({
                content: '❌ Bu işlem sadece ticket kanallarında yapılabilir!',
                ephemeral: true
            });
        }

        const ticketOwnerId = Object.keys(ticketData[guildId].activeTickets || {}).find(
            userId => ticketData[guildId].activeTickets[userId] === channel.id
        );

        if (!ticketOwnerId) {
            return interaction.reply({
                content: '❌ Bu ticketin sahibi bulunamadı!',
                ephemeral: true
            });
        }

        const ticketOwner = interaction.guild.members.cache.get(ticketOwnerId);
        const claimedBy = ticketData[guildId].claimedTickets?.[channel.id];
        const member = interaction.member;

        const isTicketOwner = member.id === ticketOwnerId;
        const isClaimedStaff = claimedBy && member.id === claimedBy;
        const hasStaffRole = staffRoleId && member.roles.cache.has(staffRoleId);

        if (!isTicketOwner && !isClaimedStaff && !hasStaffRole) {
            return interaction.reply({
                content: '❌ Bu işlemi sadece ticket sahibi veya ticketi claim eden yetkili yapabilir!',
                ephemeral: true
            });
        }

        await interaction.deferReply({ ephemeral: true });

        const guild = interaction.guild;
        const guildIcon = guild.iconURL({ dynamic: true, size: 256 });
        const guildBanner = guild.bannerURL({ dynamic: true, size: 1024 });

        if (hasStaffRole && !isTicketOwner) {
            const embed = new EmbedBuilder()
                .setTitle('🔔 Ticket Dürtme Bildirimi')
                .setDescription(`**${guild.name}** sunucusundaki ticketinizde bir güncelleme var!\n\n**Yetkili:** ${member}\n**Kanal:** ${channel}\n\n**Mesaj:** Lütfen ticketinize bakınız ve gerekli bilgileri sağlayınız. Yetkili ekibimiz size yardımcı olmak için bekliyor.`)
                .addFields(
                    { name: '📋 Ticket Bilgileri', value: `**Kanal:** ${channel.name}\n**Dürtme Zamanı:** <t:${Math.floor(Date.now() / 1000)}:F>`, inline: false },
                    { name: '⚠️ Önemli Uyarı', value: 'Bu bir otomatik bildirimdir. Lütfen ticketinize en kısa sürede dönüş yapınız.', inline: false }
                )
                .setColor('#ff9900')
                .setTimestamp();

            if (guildIcon) embed.setThumbnail(guildIcon);
            if (guildBanner) embed.setImage(guildBanner);

            try {
                await ticketOwner.send({ embeds: [embed] });
                
                const logSistemi = new LogSistemi(interaction.client);
                await logSistemi.logTicketDurt(interaction.guild, {
                    staff: member,
                    ticketOwner: ticketOwner,
                    channel: channel,
                    type: 'staff_to_owner',
                    timestamp: Date.now()
                });

                await interaction.editReply({
                    content: `✅ ${ticketOwner} kullanıcısına dürtme bildirimi gönderildi!`
                });
            } catch (error) {
                console.error('DM gönderme hatası:', error);
                await interaction.editReply({
                    content: `❌ ${ticketOwner} kullanıcısına DM gönderilemedi! (DM'leri kapalı olabilir)`
                });
            }
        }
        else if (isTicketOwner) {
            if (!claimedBy) {
                return interaction.editReply({
                    content: '❌ Bu ticket henüz kimse tarafından claim edilmemiş!'
                });
            }

            const claimedStaff = interaction.guild.members.cache.get(claimedBy);
            if (!claimedStaff) {
                return interaction.editReply({
                    content: '❌ Ticketi claim eden yetkili bulunamadı!'
                });
            }

            const embed = new EmbedBuilder()
                .setTitle('🔔 Ticket Dürtme Bildirimi')
                .setDescription(`**${guild.name}** sunucusunda claim ettiğiniz tickette yeni bir mesaj var!\n\n**Ticket Sahibi:** ${member}\n**Kanal:** ${channel}\n\n**Mesaj:** Ticket sahibi sizden yanıt bekliyor. Lütfen tickete bakınız ve gerekli yardımı sağlayınız.`)
                .addFields(
                    { name: '📋 Ticket Bilgileri', value: `**Kanal:** ${channel.name}\n**Dürtme Zamanı:** <t:${Math.floor(Date.now() / 1000)}:F>`, inline: false },
                    { name: '⚠️ Yetkili Uyarısı', value: 'Bu ticket sizin sorumluluğunuzdadır. Lütfen en kısa sürede yanıt veriniz.', inline: false }
                )
                .setColor('#0099ff')
                .setTimestamp();

            if (guildIcon) embed.setThumbnail(guildIcon);
            if (guildBanner) embed.setImage(guildBanner);

            try {
                await claimedStaff.send({ embeds: [embed] });
                
                const logSistemi = new LogSistemi(interaction.client);
                await logSistemi.logTicketDurt(interaction.guild, {
                    staff: claimedStaff,
                    ticketOwner: member,
                    channel: channel,
                    type: 'owner_to_staff',
                    timestamp: Date.now()
                });

                await interaction.editReply({
                    content: `✅ ${claimedStaff} yetkilisine dürtme bildirimi gönderildi!`
                });
            } catch (error) {
                console.error('DM gönderme hatası:', error);
                await interaction.editReply({
                    content: `❌ ${claimedStaff} yetkilisine DM gönderilemedi! (DM'leri kapalı olabilir)`
                });
            }
        }

    } catch (error) {
        console.error('Ticket dürt hatası:', error);
        await interaction.reply({
            content: '❌ Dürtme işlemi sırasında bir hata oluştu!',
            ephemeral: true
        });
    }
}

async function handleDownloadAvatar(interaction) {
    try {
        const userId = interaction.customId.split('_')[2];
        const user = await interaction.client.users.fetch(userId);
        
        const avatarURL = user.displayAvatarURL({ 
            dynamic: true, 
            size: 4096 
        });

        const downloadEmbed = new EmbedBuilder()
            .setTitle('📸 Profil Fotoğrafı İndirme')
            .setDescription(`**${user.username}** kullanıcısının profil fotoğrafı:`)
            .setImage(avatarURL)
            .setColor('#0099ff')
            .setTimestamp()
            .addFields({
                name: '🔗 İndirme Linki',
                value: `[Profil Fotoğrafını İndir](${avatarURL})`,
                inline: false
            });

        await interaction.reply({
            embeds: [downloadEmbed],
            ephemeral: true
        });

    } catch (error) {
        console.error('Avatar indirme hatası:', error);
        await interaction.reply({
            content: '❌ Profil fotoğrafı indirilemedi!',
            ephemeral: true
        });
    }
}

async function handleDownloadBanner(interaction) {
    try {
        const userId = interaction.customId.split('_')[2];
        const user = await interaction.client.users.fetch(userId);
        
        const bannerURL = user.bannerURL({ 
            dynamic: true, 
            size: 4096 
        });

        if (!bannerURL) {
            await interaction.reply({
                content: '❌ Bu kullanıcının banner\'ı bulunmuyor!',
                ephemeral: true
            });
            return;
        }

        const downloadEmbed = new EmbedBuilder()
            .setTitle('🖼️ Banner İndirme')
            .setDescription(`**${user.username}** kullanıcısının banner\'ı:`)
            .setImage(bannerURL)
            .setColor('#0099ff')
            .setTimestamp()
            .addFields({
                name: '🔗 İndirme Linki',
                value: `[Banner\'ı İndir](${bannerURL})`,
                inline: false
            });

        await interaction.reply({
            embeds: [downloadEmbed],
            ephemeral: true
        });

    } catch (error) {
        console.error('Banner indirme hatası:', error);
        await interaction.reply({
            content: '❌ Banner indirilemedi!',
            ephemeral: true
        });
    }
}

async function handleDownloadBoth(interaction) {
    try {
        const userId = interaction.customId.split('_')[2];
        const user = await interaction.client.users.fetch(userId);
        
        const avatarURL = user.displayAvatarURL({ 
            dynamic: true, 
            size: 4096 
        });
        
        const bannerURL = user.bannerURL({ 
            dynamic: true, 
            size: 4096 
        });

        if (!bannerURL) {
            await interaction.reply({
                content: '❌ Bu kullanıcının banner\'ı bulunmuyor! Sadece profil fotoğrafı indirilebilir.',
                ephemeral: true
            });
            return;
        }

        const downloadEmbed = new EmbedBuilder()
            .setTitle('💾 Tüm Görseller İndirme')
            .setDescription(`**${user.username}** kullanıcısının tüm görselleri:`)
            .setThumbnail(avatarURL)
            .setImage(bannerURL)
            .setColor('#0099ff')
            .setTimestamp()
            .addFields(
                {
                    name: '📸 Profil Fotoğrafı',
                    value: `[Profil Fotoğrafını İndir](${avatarURL})`,
                    inline: true
                },
                {
                    name: '🖼️ Banner',
                    value: `[Banner\'ı İndir](${bannerURL})`,
                    inline: true
                }
            );

        await interaction.reply({
            embeds: [downloadEmbed],
            ephemeral: true
        });

    } catch (error) {
        console.error('Tüm görseller indirme hatası:', error);
        await interaction.reply({
            content: '❌ Görseller indirilemedi!',
            ephemeral: true
        });
    }
}

async function handleGiveawaySelect(interaction) {
    console.log('Giveaway select handled in command file');
}

async function handleIcIsimRed(interaction) {
    try {
        await interaction.deferReply({ ephemeral: true });

        // CustomId'den mesaj ID'sini al: icisim_red_MESSAGE_ID
        const messageId = interaction.customId.replace('icisim_red_', '');
        const selectedReason = interaction.values[0];

        // Mesajı al
        const message = await interaction.channel.messages.fetch(messageId).catch(() => null);
        if (!message) {
            return interaction.editReply({
                content: '❌ Mesaj bulunamadı!'
            });
        }

        // Yetki kontrolü
        const { checkPermission } = require('../commands/yetkiayarla');
        if (!checkPermission(interaction, 'icisimonyared')) {
            return interaction.editReply({
                content: '❌ Bu işlemi yapmak için yetkiniz yok!'
            });
        }

        // Mesaj sahibini al
        const messageAuthor = message.author;
        if (!messageAuthor || messageAuthor.bot) {
            return interaction.editReply({
                content: '❌ Geçersiz mesaj sahibi!'
            });
        }

        // Red sebebini Türkçe'ye çevir
        let redSebebi = '';
        if (selectedReason === 'uygunsuz_isim') {
            redSebebi = 'Uygunsuz İsim';
        } else if (selectedReason === 'unlu_ismi') {
            redSebebi = 'Ünlü İsmi';
        } else {
            redSebebi = 'Bilinmeyen Sebep';
        }

        // Mesajı yanıtla
        await message.reply({
            content: `❌ IC İsim talebiniz **${redSebebi}** nedeniyle reddedilmiştir.`,
            allowedMentions: { repliedUser: true }
        });

        // Select menu'yu kaldır (mesajı güncelle)
        const updatedEmbed = new EmbedBuilder()
            .setColor(0xff0000)
            .setTitle('❌ IC İsim Reddedildi')
            .setDescription(`Red sebebi: **${redSebebi}**`)
            .setTimestamp();

        await interaction.message.edit({
            embeds: [updatedEmbed],
            components: []
        });

        // Log gönder
        try {
            const configPath = path.join(__dirname, '../../config.json');
            const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));
            
            if (config.LogSistemi.aktif && config.LogSistemi.logKanallari.icisimOnayRedLog) {
                const logChannel = interaction.guild.channels.cache.get(config.LogSistemi.logKanallari.icisimOnayRedLog);
                if (logChannel) {
                    const logEmbed = new EmbedBuilder()
                        .setColor(0xff0000)
                        .setTitle('❌ IC İsim Reddedildi')
                        .setDescription('IC İsim talebi reddedildi')
                        .addFields({
                            name: '👤 Kullanıcı',
                            value: `${messageAuthor} (${messageAuthor.id})`,
                            inline: true
                        })
                        .addFields({
                            name: '📝 İstenen İsim',
                            value: message.content.trim() || 'Yok',
                            inline: true
                        })
                        .addFields({
                            name: '❌ Red Sebebi',
                            value: redSebebi,
                            inline: true
                        })
                        .addFields({
                            name: '👮 Reddeden',
                            value: `${interaction.user} (${interaction.user.id})`,
                            inline: true
                        })
                        .addFields({
                            name: '💬 Mesaj',
                            value: `[Mesaja Git](${message.url})`,
                            inline: false
                        })
                        .setFooter({ text: 'RoveBot\'s • Log Sistemi' })
                        .setTimestamp();

                    await logChannel.send({ embeds: [logEmbed] });
                }
            }
        } catch (logError) {
            console.error('Log gönderme hatası:', logError);
        }

        await interaction.editReply({
            content: '✅ IC İsim talebi reddedildi ve kullanıcıya bildirildi.'
        });

    } catch (error) {
        console.error('IC İsim red hatası:', error);
        await interaction.editReply({
            content: '❌ Red işlemi sırasında bir hata oluştu!'
        });
    }
}

async function handleGiveawayJoin(interaction) {
    try {
        const giveawayId = interaction.customId.replace('giveaway_join_', '');
        
        const giveawayPath = path.join(__dirname, '..', 'cekilisler.json');
        if (!fs.existsSync(giveawayPath)) {
            return await interaction.reply({
                content: '❌ Çekiliş verileri bulunamadı!',
                ephemeral: true
            });
        }

        const giveaways = JSON.parse(fs.readFileSync(giveawayPath, 'utf8'));
        const giveaway = giveaways[giveawayId];

        if (!giveaway) {
            return await interaction.reply({
                content: '❌ Çekiliş bulunamadı!',
                ephemeral: true
            });
        }

        if (giveaway.status !== 'active') {
            return await interaction.reply({
                content: '❌ Bu çekiliş artık aktif değil!',
                ephemeral: true
            });
        }

        const endDate = new Date(giveaway.endDate);
        if (new Date() > endDate) {
            return await interaction.reply({
                content: '❌ Bu çekilişin süresi dolmuş!',
                ephemeral: true
            });
        }

        const userId = interaction.user.id;

        if (giveaway.participants.includes(userId)) {
            return await interaction.reply({
                content: '❌ Bu çekilişe zaten katıldınız!',
                ephemeral: true
            });
        }

        giveaway.participants.push(userId);
        giveaways[giveawayId] = giveaway;
        fs.writeFileSync(giveawayPath, JSON.stringify(giveaways, null, 2));

        try {
            const channel = interaction.guild.channels.cache.get(giveaway.channelId);
            if (channel) {
                const message = await channel.messages.fetch(giveaway.messageId);
                if (message && message.embeds.length > 0) {
                    const embed = message.embeds[0];
                    const updatedEmbed = new EmbedBuilder()
                        .setTitle(embed.title)
                        .setDescription(embed.description)
                        .setColor(embed.color)
                        .setFooter(embed.footer);

                    if (embed.timestamp) {
                        updatedEmbed.setTimestamp(new Date(embed.timestamp));
                    }

                    if (embed.thumbnail) {
                        updatedEmbed.setThumbnail(embed.thumbnail.url);
                    }
                    if (embed.image) {
                        updatedEmbed.setImage(embed.image.url);
                    }

                    updatedEmbed.addFields({
                        name: '👥 Katılımcılar',
                        value: `**${giveaway.participants.length}** kişi katıldı`,
                        inline: false
                    });

                    await message.edit({ embeds: [updatedEmbed], components: message.components });
                }
            }
        } catch (error) {
            console.error('Embed güncelleme hatası:', error);
        }

        await interaction.reply({
            content: '✅ Çekilişe başarıyla katıldınız! Bol şans! 🍀',
            ephemeral: true
        });

    } catch (error) {
        console.error('Çekilişe katılma hatası:', error);
        await interaction.reply({
            content: '❌ Çekilişe katılırken bir hata oluştu!',
            ephemeral: true
        });
    }
}

async function handleRolBilgiButton(interaction) {
    try {
        const rolbilgiCommand = interaction.client.commands.get('rolbilgi');
        if (!rolbilgiCommand) {
            await interaction.reply({
                content: '❌ Rolbilgi komutu bulunamadı!',
                ephemeral: true
            });
            return;
        }

        const customId = interaction.customId;
        
        // Sayfa butonları (Önceki/Sonraki) - önce kontrol et
        if (customId.startsWith('rolbilgi_page_')) {
            const handled = await rolbilgiCommand.handlePageButton(interaction);
            if (!handled && !interaction.replied && !interaction.deferred) {
                await interaction.reply({
                    content: '❌ Sayfa değiştirilirken bir hata oluştu!',
                    ephemeral: true
                });
            }
        }
        // Kapat butonu
        else if (customId.startsWith('rolbilgi_close_')) {
            const handled = await rolbilgiCommand.handleCloseButton(interaction);
            if (!handled && !interaction.replied && !interaction.deferred) {
                await interaction.reply({
                    content: '❌ Liste kapatılırken bir hata oluştu!',
                    ephemeral: true
                });
            }
        }
        // İlk buton (Bu Rolde Kimler Var?) - Format: rolbilgi_ROLEID_USERID
        else if (customId.startsWith('rolbilgi_')) {
            const parts = customId.split('_');
            // Eğer 3 parça varsa (rolbilgi_ROLEID_USERID) ve _page_ veya _close_ içermiyorsa
            if (parts.length === 3) {
                const handled = await rolbilgiCommand.handleButtonInteraction(interaction);
                if (!handled && !interaction.replied && !interaction.deferred) {
                    await interaction.reply({
                        content: '❌ Buton işlenirken bir hata oluştu!',
                        ephemeral: true
                    });
                }
            }
        }
    } catch (error) {
        console.error('Rolbilgi buton hatası:', error);
        if (!interaction.replied && !interaction.deferred) {
            await interaction.reply({
                content: '❌ Bir hata oluştu!',
                ephemeral: true
            });
        }
    }
}