const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const fs = require('fs');
const path = require('path');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('wlayarla')
        .setDescription('Whitelist sisteminde verilecek rolleri ayarlar')
        .addStringOption(option =>
            option.setName('roller')
                .setDescription('Verilecek rol ID\'leri (virgül ile ayırınız)')
                .setRequired(true)),

    async execute(interaction) {
        try {
            if (!checkPermission(interaction, 'wlayarla')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetkisiz Erişim')
                    .setDescription('Bu komutu kullanmak için Whitelist Ayarla yetkisine sahip olmalısınız!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const rolString = interaction.options.getString('roller');
            const rolIds = rolString.split(',').map(id => id.trim()).filter(id => id.length > 0);

            if (rolIds.length === 0) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription('En az bir rol ID belirtmelisiniz!')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            const whitelistPath = path.join(__dirname, '../whitelistSistemi.json');
            let whitelistData = {};

            if (fs.existsSync(whitelistPath)) {
                whitelistData = JSON.parse(fs.readFileSync(whitelistPath, 'utf8'));
            }

            if (!whitelistData[interaction.guild.id]) {
                whitelistData[interaction.guild.id] = {};
            }

            const gecerliRoller = [];
            const gecersizRoller = [];

            for (const rolId of rolIds) {
                const rol = interaction.guild.roles.cache.get(rolId);
                if (rol) {
                    gecerliRoller.push({
                        id: rolId,
                        name: rol.name
                    });
                } else {
                    gecersizRoller.push(rolId);
                }
            }

            if (gecerliRoller.length === 0) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription('Belirtilen rollerin hiçbiri geçerli değil!')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            whitelistData[interaction.guild.id].roller = gecerliRoller.map(r => r.id);
            fs.writeFileSync(whitelistPath, JSON.stringify(whitelistData, null, 2));

            const successEmbed = new EmbedBuilder()
                .setColor(0x00ff00)
                .setTitle('✅ Whitelist Ayarları Güncellendi')
                .setDescription('Whitelist rolleri başarıyla ayarlandı!')
                .addFields({
                    name: '✅ Geçerli Roller',
                    value: gecerliRoller.map(r => `<@&${r.id}> - ${r.name}`).join('\n'),
                    inline: false
                })
                .setFooter({ text: 'RoveBot\'s • Whitelist Sistemi' })
                .setTimestamp();

            if (gecersizRoller.length > 0) {
                successEmbed.addFields({
                    name: '⚠️ Geçersiz Roller',
                    value: gecersizRoller.join(', '),
                    inline: false
                });
                successEmbed.setDescription('Whitelist rolleri ayarlandı (bazı roller geçersiz)!');
            }

            await interaction.editReply({ embeds: [successEmbed] });

            // Log gönder
            try {
                const configPath = path.join(__dirname, '../../config.json');
                const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));
                
                if (config.LogSistemi.aktif && config.LogSistemi.logKanallari.wlAyarlaLog) {
                    const logChannel = interaction.guild.channels.cache.get(config.LogSistemi.logKanallari.wlAyarlaLog);
                    if (logChannel) {
                        const logEmbed = new EmbedBuilder()
                            .setColor(0x0099ff)
                            .setTitle('⚙️ Whitelist Ayarları Güncellendi')
                            .setDescription('Whitelist rolleri ayarlandı')
                            .addFields({
                                name: '👮 İşlem Yapan',
                                value: `${interaction.user} (${interaction.user.id})`,
                                inline: true
                            })
                            .addFields({
                                name: '✅ Ayarlanan Roller',
                                value: gecerliRoller.map(r => `<@&${r.id}>`).join('\n'),
                                inline: false
                            })
                            .setFooter({ text: 'RoveBot\'s • Log Sistemi' })
                            .setTimestamp();

                        await logChannel.send({ embeds: [logEmbed] });
                    }
                }
            } catch (logError) {
                console.error('Log gönderme hatası:', logError);
            }

        } catch (error) {
            console.error('Wlayarla komutu hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription('Whitelist ayarları güncellenirken bir hata oluştu!')
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    }
};

