const { SlashCommandBuilder, EmbedBuilder, ActionRowBuilder, StringSelectMenuBuilder, PermissionFlagsBits } = require('discord.js');
const fs = require('fs');
const path = require('path');
const { checkPermission } = require('./yetkiayarla');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('ticketsistemikur')
        .setDescription('Ticket sistemini kurar ve embed mesajını gönderir')
        .addChannelOption(option =>
            option.setName('kanal')
                .setDescription('Ticket embed mesajının gönderileceği kanal')
                .setRequired(true)
        ),

    async execute(interaction) {
        try {
            // Interaction'ı hemen defer et
            await interaction.deferReply({ ephemeral: true });

            // Yetki kontrolü - doğru parametrelerle çağır
            const hasPermission = checkPermission(interaction, 'ticketsistemi');
            if (!hasPermission) {
                return await interaction.followUp({
                    content: '❌ Bu komutu kullanmak için yetkiniz yok!',
                    ephemeral: true
                });
            }

            const channel = interaction.options.getChannel('kanal');

            // Kanal türü kontrolü
            if (channel.type !== 0) { // 0 = GUILD_TEXT
                return await interaction.followUp({
                    content: '❌ Lütfen bir metin kanalı seçin!',
                    ephemeral: true
                });
            }
            const guildId = interaction.guild.id;

            const ticketDataPath = path.join(__dirname, '..', 'ticketSistemi.json');
            let ticketData = {};
            
            if (fs.existsSync(ticketDataPath)) {
                ticketData = JSON.parse(fs.readFileSync(ticketDataPath, 'utf8'));
            }

            if (!ticketData[guildId] || !ticketData[guildId].categories || ticketData[guildId].categories.length === 0) {
                return interaction.reply({
                    content: '❌ Önce `/ticketsistemi kategori` komutu ile kategoriler oluşturmalısınız!',
                    ephemeral: true
                });
            }

            const categories = ticketData[guildId].categories;

            const embed = new EmbedBuilder()
                .setTitle('🎫 Ticket Oluştur')
                .setDescription('Merhaba! Yardıma mı ihtiyacınız var? Aşağıdaki kategorilerden birini seçerek ticket oluşturabilirsiniz. Ticket oluşturduktan sonra size özel bir kanal açılacak ve yetkili ekibimiz size yardımcı olacaktır.')
                .setColor('#00ff00')
                .setFooter({ text: 'RoveBot\'s • Ticket Sistemi' });

            if (interaction.guild.iconURL()) {
                embed.setThumbnail(interaction.guild.iconURL({ dynamic: true }));
            }
            if (interaction.guild.bannerURL()) {
                embed.setImage(interaction.guild.bannerURL({ dynamic: true, size: 1024 }));
            }

            categories.forEach(category => {
                embed.addFields({
                    name: `${category.emoji} ${category.name}`,
                    value: category.description || 'Açıklama bulunmuyor.',
                    inline: true
                });
            });

            const selectMenu = new StringSelectMenuBuilder()
                .setCustomId('ticket_category_select')
                .setPlaceholder('Bir kategori seçin...')
                .addOptions([
                    ...categories.map(category => {
                        let description = category.description || 'Açıklama yok';
                        // Discord select menu description maksimum 100 karakter olabilir
                        if (description.length > 100) {
                            description = description.substring(0, 97) + '...';
                        }
                        return {
                            label: category.name,
                            value: category.id,
                            description: description,
                            emoji: category.emoji || undefined // null yerine undefined kullan
                        };
                    }),
                    {
                        label: 'Seçeneğimi Sıfırla',
                        value: 'reset',
                        description: 'Kategori seçimini sıfırla',
                        emoji: '🧹'
                    }
                ]);

            const row = new ActionRowBuilder().addComponents(selectMenu);

            await channel.send({
                embeds: [embed],
                components: [row]
            });

            await interaction.followUp({
                content: `✅ Ticket sistemi başarıyla kuruldu! Embed mesajı ${channel} kanalına gönderildi.`,
                ephemeral: true
            });

        } catch (error) {
            console.error('Ticket sistemi kurulum hatası:', error);
            
            // Interaction'ın zaten yanıtlanıp yanıtlanmadığını kontrol et
            if (interaction.deferred || interaction.replied) {
                await interaction.followUp({
                    content: '❌ Ticket sistemi kurulurken bir hata oluştu!',
                    ephemeral: true
                });
            } else {
                await interaction.reply({
                    content: '❌ Ticket sistemi kurulurken bir hata oluştu!',
                    ephemeral: true
                });
            }
        }
    }
};