const { SlashCommandBuilder, EmbedBuilder, ActionRowBuilder, StringSelectMenuBuilder, PermissionFlagsBits } = require('discord.js');
const fs = require('fs');
const path = require('path');
const { checkPermission } = require('./yetkiayarla');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('ticketsistemi')
        .setDescription('Ticket sistemi yönetimi')
        .addSubcommandGroup(group =>
            group.setName('kategori')
                .setDescription('Ticket kategorilerini yönet')
                .addSubcommand(subcommand =>
                    subcommand.setName('ekle')
                        .setDescription('Yeni ticket kategorisi ekle')
                        .addStringOption(option =>
                            option.setName('emoji')
                                .setDescription('Kategori emojisi')
                                .setRequired(true)
                        )
                        .addStringOption(option =>
                            option.setName('ad')
                                .setDescription('Kategori adı')
                                .setRequired(true)
                        )
                        .addChannelOption(option =>
                            option.setName('kategori')
                                .setDescription('Ticketların açılacağı kategori')
                                .setRequired(true)
                        )
                        .addStringOption(option =>
                            option.setName('aciklama')
                                .setDescription('Kategori açıklaması')
                                .setRequired(false)
                        )
                )
                .addSubcommand(subcommand =>
                    subcommand.setName('sil')
                        .setDescription('Ticket kategorisi sil')
                )
                .addSubcommand(subcommand =>
                    subcommand.setName('listele')
                        .setDescription('Tüm ticket kategorilerini listele')
                )
        )
        .addSubcommandGroup(group =>
            group.setName('rol')
                .setDescription('Ticket rollerini yönet')
                .addSubcommand(subcommand =>
                    subcommand.setName('tanimla')
                        .setDescription('Ticket yetkilisi rolünü tanımla')
                        .addRoleOption(option =>
                            option.setName('rol')
                                .setDescription('Ticket yetkilisi rolü')
                                .setRequired(true)
                        )
                )
                .addSubcommand(subcommand =>
                    subcommand.setName('gor')
                        .setDescription('Mevcut ticket rolünü görüntüle')
                )
        ),

    async execute(interaction) {
        try {
            // Interaction'ı hemen defer et
            await interaction.deferReply({ ephemeral: true });

            // Yetki kontrolü - doğru parametrelerle çağır
            const hasPermission = checkPermission(interaction, 'ticketsistemi');
            if (!hasPermission) {
                return await interaction.followUp({
                    content: '❌ Bu komutu kullanmak için yetkiniz yok!',
                    ephemeral: true
                });
            }

            const ticketDataPath = path.join(__dirname, '..', 'ticketSistemi.json');
            let ticketData = {};
            
            if (fs.existsSync(ticketDataPath)) {
                ticketData = JSON.parse(fs.readFileSync(ticketDataPath, 'utf8'));
            }

            const guildId = interaction.guild.id;
            if (!ticketData[guildId]) {
                ticketData[guildId] = {
                    categories: [],
                    staffRole: null,
                    activeTickets: {}
                };
            }

            const subcommandGroup = interaction.options.getSubcommandGroup();
            const subcommand = interaction.options.getSubcommand();

            if (subcommandGroup === 'kategori') {
                if (subcommand === 'ekle') {
                    const emoji = interaction.options.getString('emoji');
                    const name = interaction.options.getString('ad');
                    const description = interaction.options.getString('aciklama') || '';
                    const category = interaction.options.getChannel('kategori');

                    const categoryId = `cat_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;

                    const newCategory = {
                        id: categoryId,
                        emoji: emoji,
                        name: name,
                        description: description,
                        categoryChannelId: category.id
                    };

                    ticketData[guildId].categories.push(newCategory);

                    fs.writeFileSync(ticketDataPath, JSON.stringify(ticketData, null, 2));

                    const embed = new EmbedBuilder()
                        .setTitle('✅ Kategori Eklendi')
                        .setDescription(`**${emoji} ${name}** kategorisi başarıyla eklendi!`)
                        .addFields(
                            { name: 'Emoji', value: emoji, inline: true },
                            { name: 'Ad', value: name, inline: true },
                            { name: 'Açıklama', value: description || 'Yok', inline: true },
                            { name: 'Kategori', value: `<#${category.id}>`, inline: true }
                        )
                        .setColor('#00ff00')
                        .setTimestamp();

                    await interaction.followUp({ embeds: [embed], ephemeral: true });

                } else if (subcommand === 'sil') {
                    const categories = ticketData[guildId].categories;
                    
                    if (categories.length === 0) {
                        return interaction.followUp({
                            content: '❌ Silinecek kategori bulunamadı!',
                            ephemeral: true
                        });
                    }

                    const selectMenu = new StringSelectMenuBuilder()
                        .setCustomId('delete_category_select')
                        .setPlaceholder('Silinecek kategoriyi seçin...')
                        .addOptions(
                            categories.map(category => ({
                                label: category.name,
                                value: category.id,
                                description: category.description || 'Açıklama yok',
                                emoji: category.emoji || undefined // null yerine undefined kullan
                            }))
                        );

                    const row = new ActionRowBuilder().addComponents(selectMenu);

                    await interaction.followUp({
                        content: '🗑️ Silinecek kategoriyi seçin:',
                        components: [row],
                        ephemeral: true
                    });

                } else if (subcommand === 'listele') {
                    const categories = ticketData[guildId].categories;
                    
                    if (categories.length === 0) {
                        return interaction.followUp({
                            content: '❌ Listelenecek kategori bulunamadı!',
                            ephemeral: true
                        });
                    }

                    const embed = new EmbedBuilder()
                        .setTitle('📋 Ticket Kategorileri')
                        .setDescription('Mevcut ticket kategorileri:')
                        .setColor('#0099ff')
                        .setTimestamp();

                    categories.forEach((category, index) => {
                        const categoryChannel = interaction.guild.channels.cache.get(category.categoryChannelId);
                        embed.addFields({
                            name: `${index + 1}. ${category.emoji} ${category.name}`,
                            value: `**Açıklama:** ${category.description || 'Yok'}\n**Kategori:** ${categoryChannel ? categoryChannel.name : 'Bulunamadı'}`,
                            inline: false
                        });
                    });

                    await interaction.followUp({ embeds: [embed], ephemeral: true });
                }

            } else if (subcommandGroup === 'rol') {
                if (subcommand === 'tanimla') {
                    const role = interaction.options.getRole('rol');

                    ticketData[guildId].staffRole = role.id;
                    fs.writeFileSync(ticketDataPath, JSON.stringify(ticketData, null, 2));

                    const embed = new EmbedBuilder()
                        .setTitle('✅ Ticket Rolü Tanımlandı')
                        .setDescription(`**${role.name}** rolü ticket yetkilisi olarak tanımlandı!`)
                        .addFields(
                            { name: 'Rol', value: `<@&${role.id}>`, inline: true },
                            { name: 'Rol ID', value: role.id, inline: true }
                        )
                        .setColor('#00ff00')
                        .setTimestamp();

                    await interaction.followUp({ embeds: [embed], ephemeral: true });

                } else if (subcommand === 'gor') {
                    const staffRoleId = ticketData[guildId].staffRole;

                    if (!staffRoleId) {
                        return await interaction.followUp({
                            content: '❌ Henüz ticket yetkilisi rolü tanımlanmamış!',
                            ephemeral: true
                        });
                    }

                    const staffRole = interaction.guild.roles.cache.get(staffRoleId);

                    const embed = new EmbedBuilder()
                        .setTitle('🎫 Ticket Yetkilisi Rolü')
                        .addFields(
                            { name: 'Rol', value: staffRole ? `<@&${staffRoleId}>` : 'Rol bulunamadı', inline: true },
                            { name: 'Rol Adı', value: staffRole ? staffRole.name : 'Bilinmiyor', inline: true },
                            { name: 'Rol ID', value: staffRoleId, inline: true }
                        )
                        .setColor('#0099ff')
                        .setTimestamp();

                    await interaction.followUp({ embeds: [embed], ephemeral: true });
                }

            } else if (subcommandGroup === 'ayarlar') {
                const subcommand = interaction.options.getSubcommand();

                if (subcommand === 'göster') {
                    const settings = ticketData[guildId].settings;

                    const embed = new EmbedBuilder()
                        .setTitle('🎫 Ticket Sistemi Ayarları')
                        .addFields(
                            { name: 'Kategori Kanalı', value: settings.categoryChannel ? `<#${settings.categoryChannel}>` : 'Ayarlanmamış', inline: true },
                            { name: 'Log Kanalı', value: settings.logChannel ? `<#${settings.logChannel}>` : 'Ayarlanmamış', inline: true },
                            { name: 'Staff Rolü', value: settings.staffRole ? `<@&${settings.staffRole}>` : 'Ayarlanmamış', inline: true }
                        )
                        .setColor('#0099ff')
                        .setTimestamp();

                    await interaction.followUp({ embeds: [embed], ephemeral: true });

                } else if (subcommand === 'sıfırla') {
                    return interaction.followUp({
                        content: '❌ Bu özellik henüz aktif değil!',
                        ephemeral: true
                    });
                }

            } else {
                const embed = new EmbedBuilder()
                    .setTitle('❌ Geçersiz Alt Komut')
                    .setDescription('Lütfen geçerli bir alt komut kullanın!')
                    .setColor('#ff0000')
                    .setTimestamp();

                await interaction.followUp({ embeds: [embed], ephemeral: true });
            }

        } catch (error) {
            console.error('Ticket sistemi hatası:', error);
            
            // Interaction'ın zaten yanıtlanıp yanıtlanmadığını kontrol et
            if (interaction.deferred || interaction.replied) {
                await interaction.followUp({
                    content: '❌ Bir hata oluştu!',
                    ephemeral: true
                });
            } else {
                await interaction.followUp({
                    content: '❌ Bir hata oluştu!',
                    ephemeral: true
                });
            }
        }
    }
};