const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits } = require('discord.js');
const fs = require('fs');
const path = require('path');
const LogSistemi = require('../logSistemi');
const { checkPermission } = require('./yetkiayarla');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('ticket')
        .setDescription('Ticket sistemi yönetimi')
        .addSubcommand(subcommand =>
            subcommand.setName('üye')
                .setDescription('Ticket kanalına üye ekle veya kaldır')
                .addStringOption(option =>
                    option.setName('eylem')
                        .setDescription('Yapılacak işlem')
                        .setRequired(true)
                        .addChoices(
                            { name: 'Ekle', value: 'ekle' },
                            { name: 'Kaldır', value: 'kaldır' }
                        )
                )
                .addUserOption(option =>
                    option.setName('üye')
                        .setDescription('İşlem yapılacak kullanıcı')
                        .setRequired(true)
                )
        ),

    async execute(interaction) {
        if (!checkPermission(interaction, 'ticketuye')) {
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Yetkisiz Erişim')
                .setDescription('Bu komutu kullanmak için Ticket Üye Yönetimi yetkisine sahip olmalısınız!')
                .setTimestamp();

            return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
        }

        const subcommand = interaction.options.getSubcommand();

        if (subcommand === 'üye') {
            await handleMemberManagement(interaction);
        }
    }
};

async function handleMemberManagement(interaction) {
    const eylem = interaction.options.getString('eylem');
    const targetUser = interaction.options.getUser('üye');
    const channel = interaction.channel;

    if (!channel.name.startsWith('🎫・ᴛɪᴄᴋᴇᴛ')) {
        return await interaction.reply({
            content: '❌ Bu komut sadece ticket kanallarında kullanılabilir!',
            ephemeral: true
        });
    }

    try {
        const currentPermissions = channel.permissionOverwrites.cache.get(targetUser.id);
        const beforePermissions = currentPermissions ? {
            viewChannel: currentPermissions.allow.has(PermissionFlagsBits.ViewChannel),
            sendMessages: currentPermissions.allow.has(PermissionFlagsBits.SendMessages)
        } : { viewChannel: false, sendMessages: false };

        if (eylem === 'ekle') {
            await channel.permissionOverwrites.edit(targetUser.id, {
                ViewChannel: true,
                SendMessages: true,
                ReadMessageHistory: true
            });

            const embed = new EmbedBuilder()
                .setColor('#00ff00')
                .setTitle('✅ Üye Eklendi')
                .setDescription(`${targetUser} kullanıcısı ticket kanalına eklendi.`)
                .addFields(
                    { name: '👤 Eklenen Kullanıcı', value: `${targetUser} (${targetUser.id})`, inline: true },
                    { name: '👮 İşlemi Yapan', value: `${interaction.user} (${interaction.user.id})`, inline: true },
                    { name: '📅 Tarih', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: false }
                )
                .setTimestamp();

            await interaction.reply({ embeds: [embed] });

            await logTicketMemberAction(interaction, targetUser, 'ekle', beforePermissions, {
                viewChannel: true,
                sendMessages: true
            });

        } else if (eylem === 'kaldır') {
            await channel.permissionOverwrites.delete(targetUser.id);

            const embed = new EmbedBuilder()
                .setColor('#ff0000')
                .setTitle('❌ Üye Kaldırıldı')
                .setDescription(`${targetUser} kullanıcısı ticket kanalından kaldırıldı.`)
                .addFields(
                    { name: '👤 Kaldırılan Kullanıcı', value: `${targetUser} (${targetUser.id})`, inline: true },
                    { name: '👮 İşlemi Yapan', value: `${interaction.user} (${interaction.user.id})`, inline: true },
                    { name: '📅 Tarih', value: `<t:${Math.floor(Date.now() / 1000)}:F>`, inline: false }
                )
                .setTimestamp();

            await interaction.reply({ embeds: [embed] });

            await logTicketMemberAction(interaction, targetUser, 'kaldır', beforePermissions, {
                viewChannel: false,
                sendMessages: false
            });
        }

    } catch (error) {
        console.error('Ticket üye yönetimi hatası:', error);
        
        const errorEmbed = new EmbedBuilder()
            .setColor('#ff0000')
            .setTitle('❌ Hata')
            .setDescription('İşlem sırasında bir hata oluştu. Lütfen tekrar deneyin.')
            .setTimestamp();

        await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
    }
}

async function logTicketMemberAction(interaction, targetUser, action, beforePermissions, afterPermissions) {
    try {
        const logData = {
            timestamp: new Date().toISOString(),
            action: action,
            channel: {
                id: interaction.channel.id,
                name: interaction.channel.name
            },
            targetUser: {
                id: targetUser.id,
                username: targetUser.username,
                displayName: targetUser.displayName
            },
            moderator: {
                id: interaction.user.id,
                username: interaction.user.username,
                displayName: interaction.user.displayName
            },
            permissions: {
                before: beforePermissions,
                after: afterPermissions
            },
            guild: {
                id: interaction.guild.id,
                name: interaction.guild.name
            }
        };

        const detailLog = {
            type: 'ticket_member_action',
            timestamp: new Date().toLocaleString('tr-TR'),
            action: action === 'ekle' ? 'Üye Eklendi' : 'Üye Kaldırıldı',
            details: logData
        };

        const statusLog = {
            type: 'ticket_permission_change',
            timestamp: new Date().toLocaleString('tr-TR'),
            action: 'İzin Durumu Değişikliği',
            details: {
                channel: logData.channel,
                user: logData.targetUser,
                permissionChange: {
                    before: beforePermissions,
                    after: afterPermissions
                },
                changedBy: logData.moderator
            }
        };

        await saveLogToFile('logguncelle.js', detailLog);
        await saveLogToFile('setuplog.js', statusLog);

        try {
            const logSistemi = new LogSistemi(interaction.client);
            await logSistemi.logTicketMemberAction(
                interaction.guild.id,
                action,
                targetUser,
                interaction.user,
                interaction.channel
            );
        } catch (logError) {
            console.error('LogSistemi kayıt hatası:', logError);
        }

    } catch (error) {
        console.error('Log kaydı hatası:', error);
    }
}

async function saveLogToFile(filename, logData) {
    try {
        const filePath = path.join(__dirname, filename);
        let existingLogs = [];

        if (fs.existsSync(filePath)) {
            const fileContent = fs.readFileSync(filePath, 'utf8');
            const lines = fileContent.split('\n');
            for (const line of lines) {
                if (line.trim().startsWith('// LOG:')) {
                    try {
                        const jsonStr = line.replace('// LOG:', '').trim();
                        existingLogs.push(JSON.parse(jsonStr));
                    } catch (e) {
                    }
                }
            }
        }

        existingLogs.push(logData);

        let fileContent = '';
        if (fs.existsSync(filePath)) {
            fileContent = fs.readFileSync(filePath, 'utf8');
        }

        const logLine = `// LOG: ${JSON.stringify(logData)}\n`;
        fileContent += logLine;

        fs.writeFileSync(filePath, fileContent);

    } catch (error) {
        console.error(`${filename} dosyasına log yazma hatası:`, error);
    }
}