const { SlashCommandBuilder, EmbedBuilder, ActionRowBuilder, ButtonBuilder, ButtonStyle, ChannelSelectMenuBuilder, ChannelType, ComponentType } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const LogSistemi = require('../logSistemi');
const fs = require('fs');
const path = require('path');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('sunucudurum')
        .setDescription('Sunucu durumunu günceller')
        .addStringOption(option =>
            option.setName('eylem')
                .setDescription('Sunucu durumu')
                .setRequired(true)
                .addChoices(
                    { name: '🟢 Aktif', value: 'Aktif' },
                    { name: '🔧 Bakım', value: 'Bakım' },
                    { name: '🔄 Restart', value: 'Restart' }
                ))
        .addStringOption(option =>
            option.setName('everyoneatilsinmi')
                .setDescription('Everyone atılsın mı?')
                .setRequired(true)
                .addChoices(
                    { name: 'Evet', value: 'evet' },
                    { name: 'Hayır', value: 'hayir' }
                )),

    async execute(interaction) {
        try {
            if (!checkPermission(interaction, 'sunucudurumkomutu')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetkisiz Erişim')
                    .setDescription('Bu komutu kullanmak için Sunucu Durum Komutu yetkisine sahip olmalısınız!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const eylem = interaction.options.getString('eylem');
            const everyoneAtilsinMi = interaction.options.getString('everyoneatilsinmi');

            const configPath = path.join(__dirname, '../../config.json');
            const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));

            if (!config.SunucuDurum || !config.SunucuDurum.aktif) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Sistem Pasif')
                    .setDescription('Sunucu durum sistemi henüz kurulmamış! Önce `/sunucudurumembed ayarla` komutunu kullanın.')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            if (!config.SunucuDurum.cfxLink) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Eksik Ayar')
                    .setDescription('CFX sunucu linki ayarlanmamış! `/sunucudurumembed ayarla` komutunu kullanarak CFX linkini ayarlayın.')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            // Durum renklerini ve açıklamalarını belirle
            let embedColor, statusIcon, statusDescription;
            
            switch (eylem) {
                case 'Aktif':
                    embedColor = '#00ff00';
                    statusIcon = '🟢';
                    statusDescription = 'Sunucu şu anda aktif ve oyunculara açık!';
                    break;
                case 'Bakım':
                    embedColor = '#ffaa00';
                    statusIcon = '🔧';
                    statusDescription = 'Sunucu bakım modunda. Lütfen daha sonra tekrar deneyin.';
                    break;
                case 'Restart':
                    embedColor = '#ff6600';
                    statusIcon = '🔄';
                    statusDescription = 'Sunucu yeniden başlatılıyor. Kısa süre sonra aktif olacak.';
                    break;
            }

            // Durumu config'e kaydet
            config.SunucuDurum.durum = eylem;
            fs.writeFileSync(configPath, JSON.stringify(config, null, 4));

            // Kanal seçim menüsü oluştur
            const channelSelect = new ChannelSelectMenuBuilder()
                .setCustomId('sunucu_durum_kanal_sec')
                .setPlaceholder('Embed\'in gönderileceği kanalı seçin...')
                .setChannelTypes(ChannelType.GuildText);

            const selectRow = new ActionRowBuilder()
                .addComponents(channelSelect);

            // Durum güncelleme onay mesajı
            const confirmEmbed = new EmbedBuilder()
                .setColor('#ffaa00')
                .setTitle('📊 Sunucu Durumu Güncellendi')
                .setDescription(`Sunucu durumu **${eylem}** olarak güncellendi!\n\nŞimdi embed'in hangi kanala gönderileceğini seçin:`)
                .addFields(
                    {
                        name: '📊 Yeni Durum',
                        value: `${statusIcon} ${eylem}`,
                        inline: true
                    },
                    {
                        name: '📢 Everyone Ping',
                        value: everyoneAtilsinMi === 'evet' ? '✅ Gönderilecek' : '❌ Gönderilmeyecek',
                        inline: true
                    }
                )
                .setFooter({ text: 'Aşağıdaki menüden kanal seçin' })
                .setTimestamp();

            const response = await interaction.editReply({ 
                embeds: [confirmEmbed], 
                components: [selectRow] 
            });

            // Kanal seçimi için collector
            const collector = response.createMessageComponentCollector({
                componentType: ComponentType.ChannelSelect,
                time: 60000
            });

            collector.on('collect', async (selectInteraction) => {
                if (selectInteraction.user.id !== interaction.user.id) {
                    return await selectInteraction.reply({
                        content: '❌ Bu menüyü sadece komutu kullanan kişi kullanabilir!',
                        ephemeral: true
                    });
                }

                await selectInteraction.deferUpdate();

                const selectedChannelId = selectInteraction.values[0];
                const selectedChannel = interaction.guild.channels.cache.get(selectedChannelId);

                if (!selectedChannel) {
                    const errorEmbed = new EmbedBuilder()
                        .setColor('#ff0000')
                        .setTitle('❌ Hata')
                        .setDescription('Seçilen kanal bulunamadı!')
                        .setTimestamp();

                    return await selectInteraction.editReply({ 
                        embeds: [errorEmbed], 
                        components: [] 
                    });
                }

                // Embed oluştur
                const serverEmbed = new EmbedBuilder()
                    .setTitle(`${statusIcon} ${interaction.guild.name}`)
                    .setDescription(statusDescription)
                    .setColor(embedColor)
                    .addFields(
                        {
                            name: '🌐 Sunucu IP',
                            value: config.SunucuDurum.ipAdresi === 'gizli' ? '🔒 GIZLI' : `\`${config.SunucuDurum.ipAdresi}\``,
                            inline: true
                        },
                        {
                            name: '📊 Durum',
                            value: `${statusIcon} ${eylem}`,
                            inline: true
                        },
                        {
                            name: '🕐 Son Güncelleme',
                            value: `<t:${Math.floor(Date.now() / 1000)}:R>`,
                            inline: true
                        }
                    )
                    .setTimestamp();

                // Sunucu profil fotoğrafını thumbnail olarak ekle
                if (interaction.guild.iconURL()) {
                    serverEmbed.setThumbnail(interaction.guild.iconURL({ dynamic: true, size: 256 }));
                }

                // Sunucu bannerını büyük resim olarak ekle
                if (interaction.guild.bannerURL()) {
                    serverEmbed.setImage(interaction.guild.bannerURL({ dynamic: true, size: 1024 }));
                }

                // Ticket odası varsa ekle
                if (config.SunucuDurum.ticketOdaId) {
                    serverEmbed.addFields({
                        name: '🎫 Destek',
                        value: `Sorun yaşıyorsanız <#${config.SunucuDurum.ticketOdaId}> kanalından destek alabilirsiniz.`,
                        inline: false
                    });
                }

                // Bağlan butonu oluştur (sadece Bakım durumu değilse)
                 let buttonRow = null;
                 if (eylem !== 'Bakım') {
                     const connectButton = new ButtonBuilder()
                         .setLabel('🎮 Sunucuya Bağlan')
                         .setURL(config.SunucuDurum.cfxLink)
                         .setStyle(ButtonStyle.Link);

                     buttonRow = new ActionRowBuilder()
                         .addComponents(connectButton);
                 }

                // Mesaj içeriğini hazırla
                let messageContent = '';
                if (everyoneAtilsinMi === 'evet') {
                    messageContent = '@everyone';
                }

                // Seçilen kanala embed gönder
                 const messageOptions = { 
                     content: messageContent, 
                     embeds: [serverEmbed]
                 };
                 
                 // Bakım durumu değilse buton ekle
                 if (buttonRow) {
                     messageOptions.components = [buttonRow];
                 }
                 
                 const sentMessage = await selectedChannel.send(messageOptions);

                // Başarı mesajı
                const successEmbed = new EmbedBuilder()
                    .setColor('#00ff00')
                    .setTitle('✅ Sunucu Durum Embed\'i Gönderildi')
                    .setDescription(`Sunucu durum embed'i başarıyla gönderildi!`)
                    .addFields(
                        {
                            name: '📊 Durum',
                            value: `${statusIcon} ${eylem}`,
                            inline: true
                        },
                        {
                            name: '📢 Everyone Ping',
                            value: everyoneAtilsinMi === 'evet' ? '✅ Gönderildi' : '❌ Gönderilmedi',
                            inline: true
                        },
                        {
                            name: '📍 Gönderildiği Kanal',
                            value: `<#${selectedChannel.id}>`,
                            inline: true
                        }
                    )
                    .setFooter({ text: 'RoveBot\'s • Sunucu Durum Sistemi' })
                    .setTimestamp();

                await selectInteraction.editReply({ 
                    embeds: [successEmbed], 
                    components: [] 
                });

                // Log kaydı
                if (interaction.client.logSistemi) {
                    const logSistemi = interaction.client.logSistemi;
                    await logSistemi.logSunucuDurum(
                        interaction.guild.id,
                        interaction.user,
                        eylem,
                        everyoneAtilsinMi === 'evet'
                    );
                }

                collector.stop();
            });

            collector.on('end', async (collected) => {
                if (collected.size === 0) {
                    const timeoutEmbed = new EmbedBuilder()
                        .setColor('#ff0000')
                        .setTitle('⏰ Zaman Aşımı')
                        .setDescription('Kanal seçimi için zaman aşımına uğradı. Lütfen komutu tekrar kullanın.')
                        .setTimestamp();

                    try {
                        await interaction.editReply({ 
                            embeds: [timeoutEmbed], 
                            components: [] 
                        });
                    } catch (error) {
                        console.log('Timeout mesajı gönderilemedi:', error.message);
                    }
                }
            });

        } catch (error) {
            console.error('Sunucudurum komutu hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setTitle('❌ Hata Oluştu')
                .setDescription(`Sunucu durumu güncellenirken bir hata oluştu:\n\`\`\`${error.message}\`\`\``)
                .setColor('#ff0000')
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed], components: [] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    }
};