const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits, ChannelType } = require('discord.js');
const LogSistemi = require('../logSistemi');
const { checkPermission } = require('./yetkiayarla');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('ses')
        .setDescription('Ses kanalı yönetim işlemleri')
        .addStringOption(option =>
            option.setName('eylem')
                .setDescription('Yapılacak işlem')
                .setRequired(true)
                .addChoices(
                    { name: 'Odadaki Herkesin Bağlantısını Kes', value: 'disconnect' },
                    { name: 'Odadaki Herkesi Mutele', value: 'mute' },
                    { name: 'Odadaki Herkesin Mutesini Aç', value: 'unmute' },
                    { name: 'Odalardaki Herkesi Tek Odaya Taşı', value: 'move' }
                ))
        .addChannelOption(option =>
            option.setName('sesodasi')
                .setDescription('İşlem yapılacak ses odası')
                .setRequired(true)
                .addChannelTypes(ChannelType.GuildVoice))
        .addChannelOption(option =>
            option.setName('hedefoda')
                .setDescription('Taşıma işlemi için hedef ses odası (sadece taşıma için)')
                .setRequired(false)
                .addChannelTypes(ChannelType.GuildVoice)),

    async execute(interaction) {
        try {
            if (!checkPermission(interaction, 'sesyonetimi')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetkisiz Erişim')
                    .setDescription('Bu komutu kullanmak için Ses Yönetimi yetkisine sahip olmalısınız!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const eylem = interaction.options.getString('eylem');
            const sesOdasi = interaction.options.getChannel('sesodasi');
            const hedefOda = interaction.options.getChannel('hedefoda');

            if (eylem === 'move' && !hedefOda) {
                const errorEmbed = new EmbedBuilder()
                    .setTitle('❌ Hata')
                    .setDescription('Taşıma işlemi için hedef ses odası belirtmelisiniz!')
                    .setColor('#ff0000')
                    .setTimestamp();
                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            const members = sesOdasi.members;
            
            if (members.size === 0) {
                const errorEmbed = new EmbedBuilder()
                    .setTitle('❌ Hata')
                    .setDescription(`${sesOdasi} ses odasında hiç üye bulunmuyor!`)
                    .setColor('#ff0000')
                    .setTimestamp();
                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            const processedMembers = [];
            const failedMembers = [];
            let actionName = '';

            for (const [memberId, member] of members) {
                try {
                    if (member.id === interaction.client.user.id) {
                        continue;
                    }

                    if (member.roles.highest.position >= interaction.member.roles.highest.position && 
                        interaction.user.id !== interaction.guild.ownerId) {
                        failedMembers.push({ member, reason: 'Yetki yetersiz' });
                        continue;
                    }

                    switch (eylem) {
                        case 'disconnect':
                            await member.voice.disconnect();
                            processedMembers.push({ member, action: 'bağlantısı kesildi' });
                            actionName = 'Bağlantı Kesme';
                            break;

                        case 'mute':
                            if (member.voice.serverMute) {
                                failedMembers.push({ member, reason: 'Zaten mute' });
                            } else {
                                await member.voice.setMute(true);
                                processedMembers.push({ member, action: 'mute edildi' });
                            }
                            actionName = 'Mute';
                            break;

                        case 'unmute':
                            if (!member.voice.serverMute) {
                                failedMembers.push({ member, reason: 'Zaten mute değil' });
                            } else {
                                await member.voice.setMute(false);
                                processedMembers.push({ member, action: 'mute açıldı' });
                            }
                            actionName = 'Mute Açma';
                            break;

                        case 'move':
                            if (member.voice.channelId === hedefOda.id) {
                                failedMembers.push({ member, reason: 'Zaten hedef odada' });
                            } else {
                                await member.voice.setChannel(hedefOda);
                                processedMembers.push({ member, action: `${hedefOda.name} odasına taşındı` });
                            }
                            actionName = 'Taşıma';
                            break;
                    }

                } catch (error) {
                    console.error(`Ses işlemi hatası (${member.user.username}):`, error);
                    failedMembers.push({ member, reason: error.message });
                }
            }

            const resultEmbed = new EmbedBuilder()
                .setTitle(`🔊 ${actionName} İşlemi Tamamlandı`)
                .setDescription(`**İşlem:** ${eylem === 'disconnect' ? 'Bağlantı Kesme' : eylem === 'mute' ? 'Mute' : eylem === 'unmute' ? 'Mute Açma' : 'Taşıma'}\n**Kaynak Oda:** ${sesOdasi}${hedefOda ? `\n**Hedef Oda:** ${hedefOda}` : ''}\n**İşlem Yapan:** ${interaction.user}\n**Tarih:** <t:${Math.floor(Date.now() / 1000)}:F>`)
                .setColor('#00ff00')
                .setTimestamp();

            if (processedMembers.length > 0) {
                const memberList = processedMembers.slice(0, 10).map(m => `• ${m.member.user.username} - ${m.action}`).join('\n');
                const remainingCount = processedMembers.length - 10;
                
                resultEmbed.addFields({
                    name: `✅ Başarılı İşlemler (${processedMembers.length})`,
                    value: memberList + (remainingCount > 0 ? `\n... ve ${remainingCount} kişi daha` : ''),
                    inline: false
                });
            }

            if (failedMembers.length > 0) {
                const failedList = failedMembers.slice(0, 5).map(m => `• ${m.member.user.username} - ${m.reason}`).join('\n');
                const remainingFailedCount = failedMembers.length - 5;
                
                resultEmbed.addFields({
                    name: `❌ Başarısız İşlemler (${failedMembers.length})`,
                    value: failedList + (remainingFailedCount > 0 ? `\n... ve ${remainingFailedCount} kişi daha` : ''),
                    inline: false
                });
            }

            resultEmbed.addFields({
                name: '📊 İstatistikler',
                value: `• **Toplam Üye:** ${members.size}\n• **Başarılı:** ${processedMembers.length}\n• **Başarısız:** ${failedMembers.length}\n• **Başarı Oranı:** ${members.size > 0 ? Math.round((processedMembers.length / members.size) * 100) : 0}%`,
                inline: false
            });

            await interaction.editReply({ embeds: [resultEmbed] });

            if (processedMembers.length > 0) {
                try {
                    const logSistemi = interaction.client.logSistemi;
                    if (logSistemi) {
                        await logSistemi.logVoiceAction({
                            action: eylem,
                            sourceChannel: sesOdasi,
                            targetChannel: hedefOda,
                            moderator: interaction.user,
                            processedMembers: processedMembers,
                            failedMembers: failedMembers,
                            guild: interaction.guild,
                            timestamp: new Date()
                        });
                    }

                    const logData = {
                        type: "voice_management",
                        timestamp: new Date().toLocaleString('tr-TR'),
                        action: actionName,
                        details: {
                            timestamp: new Date().toISOString(),
                            action: eylem,
                            sourceChannel: {
                                id: sesOdasi.id,
                                name: sesOdasi.name
                            },
                            targetChannel: hedefOda ? {
                                id: hedefOda.id,
                                name: hedefOda.name
                            } : null,
                            moderator: {
                                id: interaction.user.id,
                                username: interaction.user.username,
                                displayName: interaction.user.displayName || interaction.user.username
                            },
                            results: {
                                totalMembers: members.size,
                                successful: processedMembers.length,
                                failed: failedMembers.length,
                                successRate: members.size > 0 ? Math.round((processedMembers.length / members.size) * 100) : 0,
                                processedMembers: processedMembers.map(m => ({
                                    id: m.member.id,
                                    username: m.member.user.username,
                                    action: m.action
                                })),
                                failedMembers: failedMembers.map(m => ({
                                    id: m.member.id,
                                    username: m.member.user.username,
                                    reason: m.reason
                                }))
                            },
                            guild: {
                                id: interaction.guild.id,
                                name: interaction.guild.name
                            }
                        }
                    };

                    const logPath = path.join(__dirname, '../commands/logguncelle.js');
                    fs.appendFileSync(logPath, `\n// LOG: ${JSON.stringify(logData)}`);

                } catch (logError) {
                    console.error('Ses işlemi loglama hatası:', logError);
                }
            }

        } catch (error) {
            console.error('Ses komutu hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setTitle('❌ Hata Oluştu')
                .setDescription(`Ses işlemi sırasında bir hata oluştu:\n\`\`\`${error.message}\`\`\``)
                .setColor('#ff0000')
                .setTimestamp();

            await interaction.editReply({ embeds: [errorEmbed] });
        }
    }
};