const { SlashCommandBuilder, EmbedBuilder, ActionRowBuilder, ButtonBuilder, ButtonStyle } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const fs = require('fs');
const path = require('path');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('rolbilgi')
        .setDescription('Belirtilen rol hakkında bilgi gösterir')
        .addRoleOption(option =>
            option.setName('rol')
                .setDescription('Bilgisi gösterilecek rol')
                .setRequired(true)),

    async execute(interaction) {
        try {
            if (!checkPermission(interaction, 'rolbilgi')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetkisiz Erişim')
                    .setDescription('Bu komutu kullanmak için Rol Bilgi yetkisine sahip olmalısınız!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            const role = interaction.options.getRole('rol');
            const guild = interaction.guild;

            // Sunucu bilgilerini al
            await guild.fetch();
            
            const guildIcon = guild.iconURL({ dynamic: true, size: 4096 });
            const guildBanner = guild.bannerURL({ dynamic: true, size: 4096 });
            
            // Role sahip kullanıcıları al
            const membersWithRole = role.members.map(member => member);
            const memberCount = membersWithRole.length;

            // Rol renk kodu
            const roleColor = role.hexColor || '#000000';
            const roleColorCode = roleColor === '#000000' ? 'Renk Yok' : roleColor;

            // Embed oluştur
            const embed = new EmbedBuilder()
                .setTitle(`🎭 Rol Bilgisi: ${role.name}`)
                .setDescription(`**Rol:** <@&${role.id}>\n**Renk Kodu:** \`${roleColorCode}\`\n**Rol Adı:** ${role.name}\n**Rol ID:** \`${role.id}\`\n**Kullanıcı Sayısı:** ${memberCount} kişi`)
                .setColor(role.color || 0x0099ff)
                .setTimestamp()
                .setFooter({ text: `RoveBot's • Rol Bilgi Sistemi` });

            // Thumbnail: Sunucu profil fotoğrafı
            if (guildIcon) {
                embed.setThumbnail(guildIcon);
            }

            // Image: Sunucu banneri
            if (guildBanner) {
                embed.setImage(guildBanner);
            }

            // Rol özellikleri
            const roleFeatures = [];
            if (role.mentionable) roleFeatures.push('✅ Mentionable');
            if (role.hoist) roleFeatures.push('✅ Hoist');
            if (role.managed) roleFeatures.push('✅ Bot Rolü');
            if (role.permissions.has('Administrator')) roleFeatures.push('✅ Administrator');

            if (roleFeatures.length > 0) {
                embed.addFields({
                    name: '🔧 Rol Özellikleri',
                    value: roleFeatures.join('\n'),
                    inline: true
                });
            }

            embed.addFields({
                name: '📅 Oluşturulma Tarihi',
                value: `<t:${Math.floor(role.createdTimestamp / 1000)}:F>`,
                inline: true
            });

            // Buton oluştur
            const button = new ButtonBuilder()
                .setCustomId(`rolbilgi_${role.id}_${interaction.user.id}`)
                .setLabel('Bu Rolde Kimler Var?')
                .setStyle(ButtonStyle.Primary)
                .setEmoji('👥');

            const row = new ActionRowBuilder().addComponents(button);

            await interaction.reply({ embeds: [embed], components: [row] });

            // Log gönder
            try {
                const configPath = path.join(__dirname, '../../config.json');
                const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));
                
                if (config.LogSistemi && config.LogSistemi.aktif && config.LogSistemi.logKanallari && config.LogSistemi.logKanallari.rolBilgiLog) {
                    const logChannel = guild.channels.cache.get(config.LogSistemi.logKanallari.rolBilgiLog);
                    if (logChannel) {
                        const logEmbed = new EmbedBuilder()
                            .setColor(0x0099ff)
                            .setTitle('🎭 Rol Bilgi Komutu Kullanıldı')
                            .setDescription('Bir kullanıcı rol bilgi komutunu kullandı')
                            .addFields({
                                name: '👮 İşlem Yapan',
                                value: `${interaction.user} (${interaction.user.id})`,
                                inline: true
                            })
                            .addFields({
                                name: '🎭 Sorulan Rol',
                                value: `${role.name} (<@&${role.id}>)`,
                                inline: true
                            })
                            .addFields({
                                name: '👥 Rol Kullanıcı Sayısı',
                                value: `${memberCount} kişi`,
                                inline: true
                            })
                            .addFields({
                                name: '📍 Kanal',
                                value: `${interaction.channel} (<#${interaction.channel.id}>)`,
                                inline: true
                            })
                            .setFooter({ text: 'RoveBot\'s ✨ Log Sistemi' })
                            .setTimestamp();

                        await logChannel.send({ embeds: [logEmbed] });
                    }
                }
            } catch (logError) {
                console.error('Log gönderme hatası:', logError);
            }

        } catch (error) {
            console.error('Rolbilgi komutu hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Hata Oluştu')
                .setDescription(`Rol bilgisi alınırken bir hata oluştu:\n\`\`\`${error.message}\`\`\``)
                .setTimestamp();

            if (interaction.deferred || interaction.replied) {
                await interaction.followUp({ embeds: [errorEmbed], ephemeral: true });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    },

    // Buton interaction handler (İlk buton: Bu Rolde Kimler Var?)
    async handleButtonInteraction(interaction) {
        try {
            const customId = interaction.customId;
            // Format: rolbilgi_ROLEID_USERID
            const parts = customId.split('_');
            if (parts.length < 3) return false;

            const roleId = parts[1];
            const userId = parts[2];

            // Sadece butona tıklayan kişi görebilir
            if (interaction.user.id !== userId) {
                await interaction.reply({
                    content: '❌ Bu butonu sadece komutu kullanan kişi kullanabilir!',
                    ephemeral: true
                });
                return true;
            }

            const role = interaction.guild.roles.cache.get(roleId);
            if (!role) {
                await interaction.reply({
                    content: '❌ Rol bulunamadı!',
                    ephemeral: true
                });
                return true;
            }

            // Role sahip kullanıcıları al
            const membersWithRole = role.members.map(member => member);
            const totalMembers = membersWithRole.length;

            if (totalMembers === 0) {
                await interaction.reply({
                    content: '❌ Bu role sahip hiçbir kullanıcı bulunamadı!',
                    ephemeral: true
                });
                return true;
            }

            // Sayfalama için kullanıcıları grupla (sayfa başına 20 kullanıcı)
            const membersPerPage = 20;
            const totalPages = Math.ceil(totalMembers / membersPerPage);
            const currentPage = 0;

            const start = currentPage * membersPerPage;
            const end = Math.min(start + membersPerPage, totalMembers);
            const pageMembers = membersWithRole.slice(start, end);

            const memberList = pageMembers.map(member => 
                `${member.user} (${member.user.id})`
            ).join('\n');

            const embed = new EmbedBuilder()
                .setTitle(`👥 ${role.name} Rolüne Sahip Kullanıcılar`)
                .setDescription(memberList || 'Kullanıcı bulunamadı')
                .setFooter({ 
                    text: `Sayfa ${currentPage + 1}/${totalPages} • Toplam ${totalMembers} kullanıcı • RoveBot's` 
                })
                .setColor(role.color || 0x0099ff)
                .setTimestamp();

            const row = new ActionRowBuilder();
            
            const prevButton = new ButtonBuilder()
                .setCustomId(`rolbilgi_page_${roleId}_${userId}_${currentPage - 1}`)
                .setLabel('◀️ Önceki')
                .setStyle(ButtonStyle.Primary)
                .setDisabled(currentPage === 0);

            const nextButton = new ButtonBuilder()
                .setCustomId(`rolbilgi_page_${roleId}_${userId}_${currentPage + 1}`)
                .setLabel('Sonraki ▶️')
                .setStyle(ButtonStyle.Primary)
                .setDisabled(currentPage >= totalPages - 1);

            const closeButton = new ButtonBuilder()
                .setCustomId(`rolbilgi_close_${userId}`)
                .setLabel('❌ Kapat')
                .setStyle(ButtonStyle.Danger);

            row.addComponents(prevButton, nextButton, closeButton);

            await interaction.reply({
                embeds: [embed],
                components: [row],
                ephemeral: false
            });

            return true;
        } catch (error) {
            console.error('Rolbilgi buton interaction hatası:', error);
            if (!interaction.replied && !interaction.deferred) {
                await interaction.reply({
                    content: '❌ Bir hata oluştu!',
                    ephemeral: true
                });
            }
            return true;
        }
    },

    // Sayfa butonu handler
    async handlePageButton(interaction) {
        try {
            const customId = interaction.customId;
            if (!customId.startsWith('rolbilgi_page_')) return false;

            const parts = customId.split('_');
            if (parts.length < 6) return false;
            
            const roleId = parts[3];
            const userId = parts[4];
            const page = parseInt(parts[5]);

            // Sadece komutu kullanan kişi butonları kullanabilir
            if (interaction.user.id !== userId) {
                await interaction.reply({
                    content: '❌ Bu butonu sadece komutu kullanan kişi kullanabilir!',
                    ephemeral: true
                });
                return true;
            }

            const role = interaction.guild.roles.cache.get(roleId);
            if (!role) {
                await interaction.update({
                    content: '❌ Rol bulunamadı!',
                    embeds: [],
                    components: []
                });
                return true;
            }

            const membersWithRole = role.members.map(member => member);
            const totalMembers = membersWithRole.length;
            const membersPerPage = 20;
            const totalPages = Math.ceil(totalMembers / membersPerPage);

            if (page < 0 || page >= totalPages) {
                await interaction.reply({
                    content: '❌ Geçersiz sayfa!',
                    ephemeral: true
                });
                return true;
            }

            const embed = new EmbedBuilder()
                .setTitle(`👥 ${role.name} Rolüne Sahip Kullanıcılar`)
                .setDescription(
                    membersWithRole
                        .slice(page * membersPerPage, (page + 1) * membersPerPage)
                        .map(member => `${member.user} (${member.user.id})`)
                        .join('\n') || 'Kullanıcı bulunamadı'
                )
                .setFooter({ 
                    text: `Sayfa ${page + 1}/${totalPages} • Toplam ${totalMembers} kullanıcı • RoveBot's` 
                })
                .setColor(role.color || 0x0099ff)
                .setTimestamp();

            const row = new ActionRowBuilder();
            
            const prevButton = new ButtonBuilder()
                .setCustomId(`rolbilgi_page_${roleId}_${userId}_${page - 1}`)
                .setLabel('◀️ Önceki')
                .setStyle(ButtonStyle.Primary)
                .setDisabled(page === 0);

            const nextButton = new ButtonBuilder()
                .setCustomId(`rolbilgi_page_${roleId}_${userId}_${page + 1}`)
                .setLabel('Sonraki ▶️')
                .setStyle(ButtonStyle.Primary)
                .setDisabled(page >= totalPages - 1);

            const closeButton = new ButtonBuilder()
                .setCustomId(`rolbilgi_close_${userId}`)
                .setLabel('❌ Kapat')
                .setStyle(ButtonStyle.Danger);

            row.addComponents(prevButton, nextButton, closeButton);

            await interaction.update({
                embeds: [embed],
                components: [row]
            });

            return true;
        } catch (error) {
            console.error('Rolbilgi sayfa butonu hatası:', error);
            return true;
        }
    },

    // Kapat butonu handler
    async handleCloseButton(interaction) {
        try {
            const customId = interaction.customId;
            if (!customId.startsWith('rolbilgi_close_')) return false;

            const parts = customId.split('_');
            const userId = parts[3];

            // Sadece butona tıklayan kişi kapatabilir
            if (interaction.user.id !== userId) {
                await interaction.reply({
                    content: '❌ Bu butonu sadece komutu kullanan kişi kullanabilir!',
                    ephemeral: true
                });
                return true;
            }

            await interaction.update({
                content: '✅ Liste kapatıldı.',
                embeds: [],
                components: []
            });

            return true;
        } catch (error) {
            console.error('Rolbilgi kapat butonu hatası:', error);
            return true;
        }
    }
};

