const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits } = require('discord.js');
const LogSistemi = require('../logSistemi');
const { checkPermission } = require('./yetkiayarla');
const fs = require('fs');
const path = require('path');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('rol')
        .setDescription('Kullanıcıya rol ver veya al')
        .addStringOption(option =>
            option.setName('eylem')
                .setDescription('Yapılacak işlem')
                .setRequired(true)
                .addChoices(
                    { name: 'Ver', value: 'ver' },
                    { name: 'Al', value: 'al' }
                ))
        .addUserOption(option =>
            option.setName('kullanici')
                .setDescription('İşlem yapılacak kullanıcı')
                .setRequired(true))
        .addStringOption(option =>
            option.setName('roller')
                .setDescription('Rol ID\'leri (virgülle ayırın)')
                .setRequired(true)),

    async execute(interaction) {
        if (!checkPermission(interaction, 'rolyonetimi')) {
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Yetkisiz Erişim')
                .setDescription('Bu komutu kullanmak için Rol Yönetimi yetkisine sahip olmalısınız!')
                .setTimestamp();

            return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
        }

        const logSistemi = new LogSistemi(interaction.client);
        
        try {

            const eylem = interaction.options.getString('eylem');
            const targetUser = interaction.options.getUser('kullanici');
            const rollerString = interaction.options.getString('roller');
            
            const targetMember = await interaction.guild.members.fetch(targetUser.id);
            if (!targetMember) {
                return await interaction.reply({
                    embeds: [new EmbedBuilder()
                        .setColor('#ff0000')
                        .setTitle('❌ Hata')
                        .setDescription('Belirtilen kullanıcı bu sunucuda bulunamadı.')
                        .setTimestamp()],
                    ephemeral: true
                });
            }

            const roleIds = rollerString.split(',').map(id => id.trim());
            const roles = [];
            
            for (const roleId of roleIds) {
                const role = interaction.guild.roles.cache.get(roleId);
                if (!role) {
                    return await interaction.reply({
                        embeds: [new EmbedBuilder()
                            .setColor('#ff0000')
                            .setTitle('❌ Hata')
                            .setDescription(`Geçersiz rol ID: ${roleId}`)
                            .setTimestamp()],
                        ephemeral: true
                    });
                }
                
                if (role.position >= interaction.guild.members.me.roles.highest.position) {
                    return await interaction.reply({
                        embeds: [new EmbedBuilder()
                            .setColor('#ff0000')
                            .setTitle('❌ Hata')
                            .setDescription(`Bot'un ${role.name} rolünü yönetme yetkisi yok.`)
                            .setTimestamp()],
                        ephemeral: true
                    });
                }
                
                if (role.position >= interaction.member.roles.highest.position && interaction.user.id !== interaction.guild.ownerId) {
                    return await interaction.reply({
                        embeds: [new EmbedBuilder()
                            .setColor('#ff0000')
                            .setTitle('❌ Hata')
                            .setDescription(`${role.name} rolünü yönetme yetkiniz yok.`)
                            .setTimestamp()],
                        ephemeral: true
                    });
                }
                
                roles.push(role);
            }

            let successCount = 0;
            let failedRoles = [];
            
            for (const role of roles) {
                try {
                    if (eylem === 'ver') {
                        if (!targetMember.roles.cache.has(role.id)) {
                            await targetMember.roles.add(role);
                            successCount++;
                        }
                    } else {
                        if (targetMember.roles.cache.has(role.id)) {
                            await targetMember.roles.remove(role);
                            successCount++;
                        }
                    }
                } catch (error) {
                    failedRoles.push(role.name);
                    console.error(`Rol ${eylem} hatası:`, error);
                }
            }

            const resultEmbed = new EmbedBuilder()
                .setColor(successCount > 0 ? '#00ff00' : '#ff0000')
                .setTitle(`🎭 Rol ${eylem === 'ver' ? 'Verme' : 'Alma'} İşlemi`)
                .addFields(
                    { name: '👤 Kullanıcı', value: `<@${targetUser.id}>`, inline: true },
                    { name: '👮 Yetkili', value: `<@${interaction.user.id}>`, inline: true },
                    { name: '✅ Başarılı', value: `${successCount} rol`, inline: true }
                )
                .setTimestamp();

            if (failedRoles.length > 0) {
                resultEmbed.addFields({ name: '❌ Başarısız', value: failedRoles.join(', '), inline: false });
            }

            if (successCount > 0) {
                const successfulRoles = roles.filter(role => !failedRoles.includes(role.name));
                resultEmbed.addFields({ 
                    name: '🎭 İşlem Yapılan Roller', 
                    value: successfulRoles.map(role => `<@&${role.id}>`).join(', '), 
                    inline: false 
                });
            }

            await interaction.reply({ embeds: [resultEmbed] });

            if (successCount > 0) {
                try {
                    const successfulRoles = roles.filter(role => !failedRoles.includes(role.name));
                    await logSistemi.logRoleOperation(
                        interaction.guild.id,
                        eylem,
                        targetUser,
                        interaction.user,
                        successfulRoles,
                        interaction.channel
                    );
                } catch (logError) {
                    console.error('Rol işlemi log hatası:', logError);
                }
            }

        } catch (error) {
            console.error('Rol komutu hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor('#ff0000')
                .setTitle('❌ Hata')
                .setDescription('Rol işlemi sırasında bir hata oluştu.')
                .setTimestamp();

            if (interaction.replied || interaction.deferred) {
                await interaction.followUp({ embeds: [errorEmbed], ephemeral: true });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    }
};