const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const fs = require('fs');
const path = require('path');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('otorol')
        .setDescription('Otomatik rol sistemi yönetimi')
        .addSubcommand(subcommand =>
            subcommand
                .setName('ayarla')
                .setDescription('Otomatik rol sistemini ayarlar')
                .addRoleOption(option =>
                    option.setName('rol')
                        .setDescription('Yeni üyelere verilecek rol')
                        .setRequired(true)))
        .addSubcommand(subcommand =>
            subcommand
                .setName('durum')
                .setDescription('Otomatik rol sistemi durumu ve yönetimi')
                .addStringOption(option =>
                    option.setName('islem')
                        .setDescription('Yapılacak işlem')
                        .setRequired(true)
                        .addChoices(
                            { name: 'Aç', value: 'ac' },
                            { name: 'Kapat', value: 'kapat' },
                            { name: 'İstatistik', value: 'istatistik' },
                            { name: 'Son Üye', value: 'sonuye' }
                        ))),

    async execute(interaction) {
        try {
            if (!checkPermission(interaction, 'otorol')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetkisiz Erişim')
                    .setDescription('Bu komutu kullanmak için Otorol yetkisine sahip olmalısınız!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const subcommand = interaction.options.getSubcommand();
            const otorolPath = path.join(__dirname, '../otorolSistemi.json');

            let otorolData = {};
            if (fs.existsSync(otorolPath)) {
                otorolData = JSON.parse(fs.readFileSync(otorolPath, 'utf8'));
            }

            if (!otorolData[interaction.guild.id]) {
                otorolData[interaction.guild.id] = {
                    aktif: false,
                    rolId: null,
                    istatistikler: {
                        toplamVerilen: 0,
                        sonVerilen: null,
                        sonUye: null
                    }
                };
            }

            const guildData = otorolData[interaction.guild.id];

            if (subcommand === 'ayarla') {
                const rol = interaction.options.getRole('rol');

                const botMember = interaction.guild.members.me;
                if (rol.position >= botMember.roles.highest.position) {
                    const errorEmbed = new EmbedBuilder()
                        .setTitle('❌ Hata')
                        .setDescription(`Bu rolü veremem! **${rol.name}** rolü benim en yüksek rolümden daha yüksek veya eşit konumda.\n\nLütfen bot rolümü daha yukarı taşıyın veya daha düşük bir rol seçin.`)
                        .setColor('#ff0000')
                        .setTimestamp();

                    return await interaction.editReply({ embeds: [errorEmbed] });
                }

                guildData.rolId = rol.id;
                guildData.aktif = true;

                fs.writeFileSync(otorolPath, JSON.stringify(otorolData, null, 4));

                const successEmbed = new EmbedBuilder()
                    .setTitle('✅ Otomatik Rol Ayarlandı')
                    .setDescription(`**${rol.name}** rolü otomatik rol olarak ayarlandı!\n\n🔹 **Durum:** Aktif\n🔹 **Rol:** <@&${rol.id}>\n\n📝 **Not:** Yeni katılan üyelere otomatik olarak bu rol verilecektir.`)
                    .setColor('#00ff00')
                    .setTimestamp();

                await interaction.editReply({ embeds: [successEmbed] });

            } else if (subcommand === 'durum') {
                const islem = interaction.options.getString('islem');

                switch (islem) {
                    case 'ac':
                        if (!guildData.rolId) {
                            const errorEmbed = new EmbedBuilder()
                                .setTitle('❌ Hata')
                                .setDescription('Önce `/otorol ayarla` komutu ile bir rol ayarlamalısınız!')
                                .setColor('#ff0000')
                                .setTimestamp();

                            return await interaction.editReply({ embeds: [errorEmbed] });
                        }

                        guildData.aktif = true;
                        fs.writeFileSync(otorolPath, JSON.stringify(otorolData, null, 4));

                        const acEmbed = new EmbedBuilder()
                            .setTitle('✅ Otomatik Rol Açıldı')
                            .setDescription(`Otomatik rol sistemi **aktifleştirildi**!\n\n🔹 **Rol:** <@&${guildData.rolId}>`)
                            .setColor('#00ff00')
                            .setTimestamp();

                        await interaction.editReply({ embeds: [acEmbed] });
                        break;

                    case 'kapat':
                        guildData.aktif = false;
                        fs.writeFileSync(otorolPath, JSON.stringify(otorolData, null, 4));

                        const kapatEmbed = new EmbedBuilder()
                            .setTitle('🔒 Otomatik Rol Kapatıldı')
                            .setDescription('Otomatik rol sistemi **devre dışı bırakıldı**!')
                            .setColor('#ff6600')
                            .setTimestamp();

                        await interaction.editReply({ embeds: [kapatEmbed] });
                        break;

                    case 'istatistik':
                        const rol = guildData.rolId ? interaction.guild.roles.cache.get(guildData.rolId) : null;
                        
                        const istatistikEmbed = new EmbedBuilder()
                            .setTitle('📊 Otomatik Rol İstatistikleri')
                            .setColor(guildData.aktif ? '#00ff00' : '#ff6600')
                            .addFields(
                                {
                                    name: '🔹 Sistem Durumu',
                                    value: guildData.aktif ? '✅ Aktif' : '❌ Pasif',
                                    inline: true
                                },
                                {
                                    name: '🎭 Ayarlanan Rol',
                                    value: rol ? `<@&${rol.id}>` : '❌ Ayarlanmamış',
                                    inline: true
                                },
                                {
                                    name: '📈 Toplam Verilen',
                                    value: guildData.istatistikler.toplamVerilen.toString(),
                                    inline: true
                                },
                                {
                                    name: '⏰ Son Rol Verilen',
                                    value: guildData.istatistikler.sonVerilen ? `<t:${Math.floor(guildData.istatistikler.sonVerilen / 1000)}:R>` : 'Henüz verilmedi',
                                    inline: true
                                },
                                {
                                    name: '👤 Son Üye',
                                    value: guildData.istatistikler.sonUye ? `<@${guildData.istatistikler.sonUye}>` : 'Henüz yok',
                                    inline: true
                                }
                            )
                            .setTimestamp();

                        await interaction.editReply({ embeds: [istatistikEmbed] });
                        break;

                    case 'sonuye':
                        if (!guildData.istatistikler.sonUye) {
                            const sonuyeErrorEmbed = new EmbedBuilder()
                                .setTitle('❌ Bilgi Yok')
                                .setDescription('Henüz hiç üye katılmamış veya rol verilmemiş.')
                                .setColor('#ff6600')
                                .setTimestamp();

                            return await interaction.editReply({ embeds: [sonuyeErrorEmbed] });
                        }

                        const sonUye = await interaction.guild.members.fetch(guildData.istatistikler.sonUye).catch(() => null);
                        const rol2 = guildData.rolId ? interaction.guild.roles.cache.get(guildData.rolId) : null;

                        const sonuyeEmbed = new EmbedBuilder()
                            .setTitle('👤 Son Katılan Üye')
                            .setColor('#0099ff')
                            .addFields(
                                {
                                    name: '👤 Üye',
                                    value: sonUye ? `${sonUye.user.tag} (<@${sonUye.id}>)` : `<@${guildData.istatistikler.sonUye}> (Sunucudan ayrılmış)`,
                                    inline: false
                                },
                                {
                                    name: '🎭 Verilen Rol',
                                    value: rol2 ? `<@&${rol2.id}>` : 'Rol bulunamadı',
                                    inline: true
                                },
                                {
                                    name: '⏰ Katılma Zamanı',
                                    value: guildData.istatistikler.sonVerilen ? `<t:${Math.floor(guildData.istatistikler.sonVerilen / 1000)}:F>` : 'Bilinmiyor',
                                    inline: true
                                }
                            )
                            .setTimestamp();

                        if (sonUye && sonUye.user.displayAvatarURL()) {
                            sonuyeEmbed.setThumbnail(sonUye.user.displayAvatarURL());
                        }

                        await interaction.editReply({ embeds: [sonuyeEmbed] });
                        break;
                }
            }

        } catch (error) {
            console.error('Otorol komutu hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setTitle('❌ Hata Oluştu')
                .setDescription(`Komut çalıştırılırken bir hata oluştu:\n\`\`\`${error.message}\`\`\``)
                .setColor('#ff0000')
                .setTimestamp();

            await interaction.editReply({ embeds: [errorEmbed] });
        }
    }
};