const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const LogSistemi = require('../logSistemi');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('nuke')
        .setDescription('Belirtilen kanalı siler ve aynı ayarlarla yeniden oluşturur')
        .addChannelOption(option =>
            option.setName('kanal')
                .setDescription('Nuke edilecek kanal (boş bırakılırsa mevcut kanal)')
                .setRequired(false)
        ),

    async execute(interaction) {
        try {
            // Interaction'ı hemen defer et
            await interaction.deferReply({ ephemeral: true });

            // Yetki kontrolü
            const hasPermission = checkPermission(interaction, 'nuke');
            if (!hasPermission) {
                return await interaction.followUp({
                    content: '❌ Bu komutu kullanmak için yetkiniz yok!',
                    ephemeral: true
                });
            }

            // Kanal seçimi - eğer belirtilmemişse mevcut kanal
            const targetChannel = interaction.options.getChannel('kanal') || interaction.channel;

            // Kanal türü kontrolü (sadece metin ve ses kanalları)
            if (![0, 2, 4, 5, 13, 15].includes(targetChannel.type)) {
                return await interaction.followUp({
                    content: '❌ Bu kanal türü nuke edilemez! Sadece metin, ses, kategori, forum ve stage kanalları desteklenir.',
                    ephemeral: true
                });
            }

            // Bot'un gerekli izinleri var mı kontrol et
            const botMember = interaction.guild.members.me;
            if (!botMember.permissions.has([PermissionFlagsBits.ManageChannels])) {
                return await interaction.followUp({
                    content: '❌ Bu komutu kullanabilmem için "Kanalları Yönet" iznine ihtiyacım var!',
                    ephemeral: true
                });
            }

            // Kanal bilgilerini kaydet
            const channelData = {
                name: targetChannel.name,
                type: targetChannel.type,
                topic: targetChannel.topic,
                nsfw: targetChannel.nsfw,
                bitrate: targetChannel.bitrate,
                userLimit: targetChannel.userLimit,
                rateLimitPerUser: targetChannel.rateLimitPerUser,
                position: targetChannel.position,
                parent: targetChannel.parent,
                permissionOverwrites: targetChannel.permissionOverwrites.cache.map(overwrite => ({
                    id: overwrite.id,
                    type: overwrite.type,
                    allow: overwrite.allow.bitfield.toString(),
                    deny: overwrite.deny.bitfield.toString()
                }))
            };

            // Özel kanal ayarları (forum kanalları için)
            if (targetChannel.type === 15) { // Forum channel
                channelData.defaultReactionEmoji = targetChannel.defaultReactionEmoji;
                channelData.defaultThreadRateLimitPerUser = targetChannel.defaultThreadRateLimitPerUser;
                channelData.defaultSortOrder = targetChannel.defaultSortOrder;
                channelData.defaultForumLayout = targetChannel.defaultForumLayout;
                channelData.availableTags = targetChannel.availableTags;
            }

            // Stage kanalları için
            if (targetChannel.type === 13) { // Stage channel
                channelData.rtcRegion = targetChannel.rtcRegion;
            }

            const oldChannelName = targetChannel.name;
            const oldChannelId = targetChannel.id;
            const isCurrentChannel = targetChannel.id === interaction.channel.id;

            // Eski kanalı sil
            await targetChannel.delete('Nuke komutu ile silinmiştir');

            // Yeni kanalı oluştur
            const newChannel = await interaction.guild.channels.create({
                name: channelData.name,
                type: channelData.type,
                topic: channelData.topic,
                nsfw: channelData.nsfw,
                bitrate: channelData.bitrate,
                userLimit: channelData.userLimit,
                rateLimitPerUser: channelData.rateLimitPerUser,
                position: channelData.position,
                parent: channelData.parent,
                permissionOverwrites: channelData.permissionOverwrites.map(overwrite => ({
                    id: overwrite.id,
                    type: overwrite.type,
                    allow: BigInt(overwrite.allow),
                    deny: BigInt(overwrite.deny)
                })),
                // Forum kanalları için özel ayarlar
                ...(channelData.type === 15 && {
                    defaultReactionEmoji: channelData.defaultReactionEmoji,
                    defaultThreadRateLimitPerUser: channelData.defaultThreadRateLimitPerUser,
                    defaultSortOrder: channelData.defaultSortOrder,
                    defaultForumLayout: channelData.defaultForumLayout,
                    availableTags: channelData.availableTags
                }),
                // Stage kanalları için özel ayarlar
                ...(channelData.type === 13 && {
                    rtcRegion: channelData.rtcRegion
                })
            });

            // Log sistemi
            try {
                const logSistemi = new LogSistemi(interaction.client);
                await logSistemi.logNuke(interaction.guild.id, interaction.user, {
                    name: oldChannelName,
                    id: oldChannelId,
                    topic: channelData.topic
                }, newChannel);
            } catch (logError) {
                console.error('Nuke log hatası:', logError);
            }

            // Başarı mesajı
            const embed = new EmbedBuilder()
                .setTitle('💥 Kanal Nuke Edildi!')
                .setDescription(`**${oldChannelName}** kanalı başarıyla nuke edildi ve yeniden oluşturuldu!`)
                .addFields(
                    { name: '🗑️ Eski Kanal ID', value: oldChannelId, inline: true },
                    { name: '🆕 Yeni Kanal', value: `<#${newChannel.id}>`, inline: true },
                    { name: '🆔 Yeni Kanal ID', value: newChannel.id, inline: true },
                    { name: '📋 Kopyalanan Özellikler', value: '• İzinler\n• Kanal Adı\n• Kanal Türü\n• Konu/Açıklama\n• NSFW Ayarı\n• Yavaş Mod\n• Pozisyon\n• Kategori', inline: false }
                )
                .setColor('#ff6b6b')
                .setTimestamp()
                .setFooter({ text: `Nuke eden: ${interaction.user.tag}`, iconURL: interaction.user.displayAvatarURL() });

            // Eğer nuke edilen kanal mevcut kanaldı, yeni kanala mesaj gönder
            if (isCurrentChannel) {
                await newChannel.send({ embeds: [embed] });
                
                // Kullanıcıya özel mesaj gönder
                try {
                    await interaction.user.send({
                        content: `✅ **${oldChannelName}** kanalı başarıyla nuke edildi! Yeni kanal: <#${newChannel.id}>`,
                        embeds: [embed]
                    });
                } catch (error) {
                    console.log('DM gönderilemedi:', error.message);
                }
            } else {
                // Farklı bir kanal nuke edildiyse, mevcut kanala yanıt ver
                await interaction.followUp({ embeds: [embed], ephemeral: true });
            }

        } catch (error) {
            console.error('Nuke komutu hatası:', error);
            
            try {
                if (interaction.deferred || interaction.replied) {
                    await interaction.followUp({
                        content: '❌ Kanal nuke edilirken bir hata oluştu!',
                        ephemeral: true
                    });
                } else {
                    await interaction.reply({
                        content: '❌ Kanal nuke edilirken bir hata oluştu!',
                        ephemeral: true
                    });
                }
            } catch (followUpError) {
                console.error('Hata mesajı gönderilirken sorun oluştu:', followUpError);
            }
        }
    }
};