const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits } = require('discord.js');
const LogSistemi = require('../logSistemi');
const { checkPermission } = require('./yetkiayarla');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('mesajsil')
        .setDescription('Mesajları sil')
        .addStringOption(option =>
            option.setName('eylem')
                .setDescription('Yapılacak işlem')
                .setRequired(true)
                .addChoices(
                    { name: 'Kullanıcı Mesajı Sil', value: 'kullanici' },
                    { name: 'Odadaki Mesajları Sil', value: 'oda' }
                ))
        .addIntegerOption(option =>
            option.setName('adet')
                .setDescription('Silinecek mesaj sayısı (1-100)')
                .setRequired(true)
                .setMinValue(1)
                .setMaxValue(100))
        .addUserOption(option =>
            option.setName('kullanici')
                .setDescription('Mesajları silinecek kullanıcı (sadece kullanıcı mesajı sil için)')
                .setRequired(false)),

    async execute(interaction) {
        if (!checkPermission(interaction, 'mesajsil')) {
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Yetkisiz Erişim')
                .setDescription('Bu komutu kullanmak için Mesaj Silme yetkisine sahip olmalısınız!')
                .setTimestamp();

            return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
        }

        const logSistemi = new LogSistemi(interaction.client);
        
        try {
            await interaction.deferReply({ ephemeral: true });

            const eylem = interaction.options.getString('eylem');
            const adet = interaction.options.getInteger('adet');
            const targetUser = interaction.options.getUser('kullanici');

            if (eylem === 'kullanici' && !targetUser) {
                return await interaction.editReply({
                    embeds: [new EmbedBuilder()
                        .setColor('#ff0000')
                        .setTitle('❌ Hata')
                        .setDescription('Kullanıcı mesajı silmek için bir kullanıcı belirtmelisiniz.')
                        .setTimestamp()]
                });
            }

            let deletedCount = 0;
            const channel = interaction.channel;

            if (eylem === 'oda') {
                const messages = await channel.messages.fetch({ limit: adet });
                
                const now = Date.now();
                const twoWeeksAgo = now - (14 * 24 * 60 * 60 * 1000);
                
                const recentMessages = messages.filter(msg => msg.createdTimestamp > twoWeeksAgo);
                const oldMessages = messages.filter(msg => msg.createdTimestamp <= twoWeeksAgo);

                if (recentMessages.size > 0) {
                    if (recentMessages.size === 1) {
                        await recentMessages.first().delete();
                        deletedCount += 1;
                    } else {
                        const deleted = await channel.bulkDelete(recentMessages, true);
                        deletedCount += deleted.size;
                    }
                }

                for (const message of oldMessages.values()) {
                    try {
                        await message.delete();
                        deletedCount++;
                        await new Promise(resolve => setTimeout(resolve, 100));
                    } catch (error) {
                        console.error('Eski mesaj silme hatası:', error);
                    }
                }

            } else if (eylem === 'kullanici') {
                const messages = await channel.messages.fetch({ limit: 100 });
                const userMessages = messages.filter(msg => msg.author.id === targetUser.id).first(adet);
                
                const now = Date.now();
                const twoWeeksAgo = now - (14 * 24 * 60 * 60 * 1000);
                
                for (const message of userMessages) {
                    try {
                        if (message.createdTimestamp > twoWeeksAgo) {
                            await message.delete();
                        } else {
                            await message.delete();
                        }
                        deletedCount++;
                        await new Promise(resolve => setTimeout(resolve, 100));
                    } catch (error) {
                        console.error('Kullanıcı mesajı silme hatası:', error);
                    }
                }
            }

            const resultEmbed = new EmbedBuilder()
                .setColor('#00ff00')
                .setTitle('🗑️ Mesaj Silme İşlemi')
                .addFields(
                    { name: '👮 Yetkili', value: `<@${interaction.user.id}>`, inline: true },
                    { name: '📍 Kanal', value: `<#${channel.id}>`, inline: true },
                    { name: '🔢 Silinen Mesaj', value: `${deletedCount} mesaj`, inline: true },
                    { name: '⚡ İşlem Türü', value: eylem === 'kullanici' ? 'Kullanıcı mesajları' : 'Kanal mesajları', inline: false }
                )
                .setTimestamp();

            if (targetUser) {
                resultEmbed.addFields({ name: '👤 Hedef Kullanıcı', value: `<@${targetUser.id}>`, inline: false });
            }

            await interaction.editReply({ embeds: [resultEmbed] });

            if (deletedCount > 0) {
                try {
                    await logSistemi.logMessageDeleteOperation(
                        interaction.guild.id,
                        interaction.user,
                        channel,
                        deletedCount,
                        targetUser
                    );
                } catch (logError) {
                    console.error('Mesaj silme işlemi log hatası:', logError);
                }
            }

        } catch (error) {
            console.error('Mesaj silme komutu hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor('#ff0000')
                .setTitle('❌ Hata')
                .setDescription('Mesaj silme işlemi sırasında bir hata oluştu.')
                .setTimestamp();

            if (interaction.replied || interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    }
};