const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits } = require('discord.js');
const fs = require('fs');
const path = require('path');
const LogSistemi = require('../logSistemi');

function checkPermission(interaction, requiredPermission) {
    // Null/undefined kontrolleri ekle
    if (!interaction || !interaction.guild || !interaction.member) {
        console.error('herkestenrolal.js checkPermission: interaction, guild veya member undefined');
        return false;
    }

    const yetkiPath = path.join(__dirname, '../yetkiSistemi.json');
    
    if (!fs.existsSync(yetkiPath)) {
        return interaction.member.permissions.has(PermissionFlagsBits.Administrator);
    }

    try {
        const yetkiData = JSON.parse(fs.readFileSync(yetkiPath, 'utf8'));
        const guildData = yetkiData[interaction.guild.id];

        if (!guildData || !guildData[requiredPermission]) {
            return interaction.member.permissions.has(PermissionFlagsBits.Administrator);
        }

        return interaction.member.roles.cache.has(guildData[requiredPermission]) || 
               interaction.member.permissions.has(PermissionFlagsBits.Administrator);
    } catch (error) {
        console.error('herkestenrolal.js checkPermission JSON parse hatası:', error);
        return interaction.member.permissions.has(PermissionFlagsBits.Administrator);
    }
}

module.exports = {
    data: new SlashCommandBuilder()
        .setName('herkestenrolal')
        .setDescription('Sunucudaki tüm üyelerden belirtilen rolü alır')
        .addRoleOption(option =>
            option.setName('rol')
                .setDescription('Alınacak rol')
                .setRequired(true)),

    async execute(interaction) {
        try {
            if (!checkPermission(interaction, 'herkestenrolal')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetkisiz Erişim')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const targetRole = interaction.options.getRole('rol');
            const guild = interaction.guild;

            if (targetRole.position >= interaction.member.roles.highest.position && !interaction.member.permissions.has(PermissionFlagsBits.Administrator)) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Rol Hiyerarşisi Hatası')
                    .setDescription('Bu rolü almak için yeterli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            if (targetRole.position >= guild.members.me.roles.highest.position) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Bot Yetki Hatası')
                    .setDescription('Bu rolü almak için botun yeterli yetkisi yok!')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            if (targetRole.id === guild.roles.everyone.id) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Geçersiz Rol')
                    .setDescription('@everyone rolü alınamaz!')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            await guild.members.fetch();
            const members = guild.members.cache.filter(member => !member.user.bot && member.roles.cache.has(targetRole.id));
            
            if (members.size === 0) {
                const noMembersEmbed = new EmbedBuilder()
                    .setColor(0xffaa00)
                    .setTitle('⚠️ Rol Bulunamadı')
                    .setDescription(`**${targetRole.name}** rolüne sahip hiçbir üye bulunamadı!`)
                    .setTimestamp();

                return await interaction.editReply({ embeds: [noMembersEmbed] });
            }

            let successCount = 0;
            let failCount = 0;

            const progressEmbed = new EmbedBuilder()
                .setColor(0xffaa00)
                .setTitle('🔄 Rol Alma İşlemi Başlatıldı')
                .setDescription(`**${targetRole.name}** rolü **${members.size}** üyeden alınıyor...\n\nBu işlem biraz zaman alabilir.`)
                .setTimestamp();

            await interaction.editReply({ embeds: [progressEmbed] });

            for (const [memberId, member] of members) {
                try {
                    await member.roles.remove(targetRole);
                    successCount++;
                    
                    if ((successCount + failCount) % 10 === 0) {
                        const updateEmbed = new EmbedBuilder()
                            .setColor(0xffaa00)
                            .setTitle('🔄 Rol Alma İşlemi Devam Ediyor')
                            .setDescription(`**İşlenen:** ${successCount + failCount}/${members.size}\n**Başarılı:** ${successCount}\n**Başarısız:** ${failCount}`)
                            .setTimestamp();

                        await interaction.editReply({ embeds: [updateEmbed] });
                    }

                } catch (error) {
                    console.error(`${member.user.tag} üyesinden rol alınamadı:`, error);
                    failCount++;
                }
            }

            const resultEmbed = new EmbedBuilder()
                .setColor(successCount > 0 ? 0x00ff00 : 0xff0000)
                .setTitle('✅ Rol Alma İşlemi Tamamlandı')
                .setDescription(`**${targetRole.name}** rolü alma işlemi tamamlandı!`)
                .addFields(
                    {
                        name: '📊 İstatistikler',
                        value: `• **Rol Sahibi Üye:** ${members.size}\n• **Başarılı:** ${successCount}\n• **Başarısız:** ${failCount}`,
                        inline: false
                    },
                    {
                        name: '👤 İşlemi Yapan',
                        value: `<@${interaction.user.id}>`,
                        inline: true
                    },
                    {
                        name: '🎭 Alınan Rol',
                        value: `<@&${targetRole.id}>`,
                        inline: true
                    }
                )
                .setFooter({ text: `RoveBot\'s • Rol Yönetimi` })
                .setTimestamp();

            await interaction.editReply({ embeds: [resultEmbed] });

            try {
                const logSistemi = new LogSistemi(interaction.client);
                await logSistemi.logRoleManagement({
                    type: 'HERKESTENROLAL',
                    executor: interaction.user,
                    role: targetRole,
                    affectedCount: successCount,
                    totalCount: members.size,
                    guild: guild
                });
            } catch (logError) {
                console.error('Rol alma işlemi loglanamadı:', logError);
            }

        } catch (error) {
            console.error('HerkestenRolAl komutu hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription('Rol alma işlemi sırasında bir hata oluştu!')
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    }
};