const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits } = require('discord.js');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('davet')
        .setDescription('Davet istatistiklerini görüntüle')
        .addSubcommand(subcommand =>
            subcommand
                .setName('istatistik')
                .setDescription('Kendi davet istatistiklerinizi görüntüleyin')
        )
        .addSubcommand(subcommand =>
            subcommand
                .setName('kullanici')
                .setDescription('Belirli bir kullanıcının davet istatistiklerini görüntüleyin')
                .addUserOption(option =>
                    option
                        .setName('kullanici')
                        .setDescription('İstatistikleri görüntülenecek kullanıcı')
                        .setRequired(true)
                )
        )
        .addSubcommand(subcommand =>
            subcommand
                .setName('siralamasi')
                .setDescription('Sunucunun en çok davet eden kullanıcılarını görüntüleyin')
                .addIntegerOption(option =>
                    option
                        .setName('limit')
                        .setDescription('Gösterilecek kullanıcı sayısı (varsayılan: 10)')
                        .setMinValue(1)
                        .setMaxValue(25)
                )
        ),

    async execute(interaction) {
        try {
            const subcommand = interaction.options.getSubcommand();
            const inviteTracker = interaction.client.inviteTracker;

            if (!inviteTracker) {
                return await interaction.reply({
                    content: '❌ Davet takip sistemi aktif değil!',
                    ephemeral: true
                });
            }

            switch (subcommand) {
                case 'istatistik':
                    await this.handleUserStats(interaction, interaction.user.id, inviteTracker);
                    break;

                case 'kullanici':
                    const targetUser = interaction.options.getUser('kullanici');
                    await this.handleUserStats(interaction, targetUser.id, inviteTracker, targetUser);
                    break;

                case 'siralamasi':
                    const limit = interaction.options.getInteger('limit') || 10;
                    await this.handleLeaderboard(interaction, inviteTracker, limit);
                    break;
            }

        } catch (error) {
            console.error('Davet komutu hatası:', error);
            await interaction.reply({
                content: '❌ Komut çalıştırılırken bir hata oluştu!',
                ephemeral: true
            });
        }
    },

    async handleUserStats(interaction, userId, inviteTracker, targetUser = null) {
        const user = targetUser || interaction.user;
        const stats = inviteTracker.getInviteStats(interaction.guild.id, userId);

        if (!stats || stats.totalInvites === 0) {
            return await interaction.reply({
                content: `📊 ${user.tag} henüz hiç davet oluşturmamış veya kullanılmamış.`,
                ephemeral: true
            });
        }

        const embed = new EmbedBuilder()
            .setTitle('📊 Davet İstatistikleri')
            .setDescription(`**Kullanıcı:** ${user.tag} (${user.id})`)
            .setThumbnail(user.displayAvatarURL({ dynamic: true }))
            .addFields(
                {
                    name: '🎫 Toplam Davet',
                    value: stats.totalInvites.toString(),
                    inline: true
                },
                {
                    name: '📋 Aktif Davet Kodu',
                    value: stats.activeInvites.toString(),
                    inline: true
                },
                {
                    name: '📈 Ortalama Kullanım',
                    value: stats.activeInvites > 0 ? 
                        (stats.totalInvites / stats.activeInvites).toFixed(1) : '0',
                    inline: true
                }
            )
            .setColor('#00aaff')
            .setTimestamp()
            .setFooter({
                text: `RoveBot\'s • Davet Sistemi`,
                iconURL: interaction.guild.iconURL({ dynamic: true })
            });

        await interaction.reply({ embeds: [embed] });
    },

    async handleLeaderboard(interaction, inviteTracker, limit) {
        const topInviters = inviteTracker.getTopInviters(interaction.guild.id, limit);

        if (topInviters.length === 0) {
            return await interaction.reply({
                content: '📊 Henüz hiç davet istatistiği bulunmuyor.',
                ephemeral: true
            });
        }

        const embed = new EmbedBuilder()
            .setTitle('🏆 Davet Sıralaması')
            .setDescription(`**${interaction.guild.name}** sunucusunun en çok davet eden kullanıcıları`)
            .setColor('#ffd700')
            .setTimestamp()
            .setFooter({
                text: `RoveBot\'s • Davet Sistemi`,
                iconURL: interaction.guild.iconURL({ dynamic: true })
            });

        let description = '';
        const medals = ['🥇', '🥈', '🥉'];

        topInviters.forEach((inviter, index) => {
            const medal = index < 3 ? medals[index] : `**${index + 1}.**`;
            description += `${medal} **${inviter.user.tag}**\n`;
            description += `   └ 🎫 ${inviter.invites} davet • 📋 ${inviter.codes} kod\n\n`;
        });

        embed.setDescription(description);

        await interaction.reply({ embeds: [embed] });
    }
};