const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits, ActionRowBuilder, ButtonBuilder, ButtonStyle, ChannelType } = require('discord.js');
const fs = require('fs');
const path = require('path');
const LogSistemi = require('../logSistemi');

function checkPermission(interaction, requiredPermission) {
    // Null/undefined kontrolleri ekle
    if (!interaction || !interaction.guild || !interaction.member) {
        console.error('cekilis.js checkPermission: interaction, guild veya member undefined');
        return false;
    }

    const yetkiPath = path.join(__dirname, '../yetkiSistemi.json');
    
    if (!fs.existsSync(yetkiPath)) {
        return interaction.member.permissions.has(PermissionFlagsBits.Administrator);
    }

    try {
        const yetkiData = JSON.parse(fs.readFileSync(yetkiPath, 'utf8'));
        const guildData = yetkiData[interaction.guild.id];

        if (!guildData || !guildData[requiredPermission]) {
            return interaction.member.permissions.has(PermissionFlagsBits.Administrator);
        }

        return interaction.member.roles.cache.has(guildData[requiredPermission]) || 
               interaction.member.permissions.has(PermissionFlagsBits.Administrator);
    } catch (error) {
        console.error('cekilis.js checkPermission JSON parse hatası:', error);
        return interaction.member.permissions.has(PermissionFlagsBits.Administrator);
    }
}

function loadGiveaways() {
    const giveawayPath = path.join(__dirname, '../cekilisler.json');
    if (!fs.existsSync(giveawayPath)) {
        fs.writeFileSync(giveawayPath, JSON.stringify({}, null, 2));
        return {};
    }
    return JSON.parse(fs.readFileSync(giveawayPath, 'utf8'));
}

function saveGiveaways(data) {
    const giveawayPath = path.join(__dirname, '../cekilisler.json');
    fs.writeFileSync(giveawayPath, JSON.stringify(data, null, 2));
}

function generateGiveawayId() {
    return Date.now().toString(36) + Math.random().toString(36).substr(2);
}

module.exports = {
    data: new SlashCommandBuilder()
        .setName('cekilis')
        .setDescription('Yeni bir çekiliş oluşturur')
        .addChannelOption(option =>
            option.setName('kanal')
                .setDescription('Çekiliş kanalı')
                .addChannelTypes(ChannelType.GuildText)
                .setRequired(true))
        .addStringOption(option =>
            option.setName('baslik')
                .setDescription('Çekiliş başlığı')
                .setRequired(true))
        .addStringOption(option =>
            option.setName('aciklama')
                .setDescription('Çekiliş açıklaması')
                .setRequired(true))
        .addIntegerOption(option =>
            option.setName('kazanan_sayisi')
                .setDescription('Çekiliş kazanan kişi sayısı')
                .setMinValue(1)
                .setMaxValue(50)
                .setRequired(true))
        .addStringOption(option =>
            option.setName('odul')
                .setDescription('Çekiliş ödülü')
                .setRequired(true))
        .addIntegerOption(option =>
            option.setName('gun_sayisi')
                .setDescription('Çekiliş sonuçlanma tarihi (gün sayısı)')
                .setMinValue(1)
                .setMaxValue(30)
                .setRequired(true)),

    async execute(interaction) {
        try {
            if (!checkPermission(interaction, 'cekilissorumlusu')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetkisiz Erişim')
                    .setDescription('Bu komutu kullanmak için Çekiliş Sorumlusu yetkisine sahip olmalısınız!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply({ ephemeral: true });

            const channel = interaction.options.getChannel('kanal');
            const title = interaction.options.getString('baslik');
            const description = interaction.options.getString('aciklama');
            const winnerCount = interaction.options.getInteger('kazanan_sayisi');
            const prize = interaction.options.getString('odul');
            const days = interaction.options.getInteger('gun_sayisi');

            if (!channel.permissionsFor(interaction.guild.members.me).has([PermissionFlagsBits.SendMessages, PermissionFlagsBits.ViewChannel])) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Kanal Yetki Hatası')
                    .setDescription('Belirtilen kanala mesaj gönderme yetkim yok!')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            const endDate = new Date();
            endDate.setDate(endDate.getDate() + days);

            const giveawayId = generateGiveawayId();

            const giveawayEmbed = new EmbedBuilder()
                .setTitle(`🎉 ${title}`)
                .setDescription(`${description}\n\n🎁 **Ödül:** ${prize}\n👥 **Kazanan Sayısı:** ${winnerCount}\n⏰ **Bitiş Tarihi:** <t:${Math.floor(endDate.getTime() / 1000)}:F>\n🟢 **Durum:** Aktif\n👤 **Oluşturan:** <@${interaction.user.id}>\n\n**Katılmak için aşağıdaki butona tıklayın!**`)
                .addFields({
                    name: '👥 Katılımcılar',
                    value: '**0** kişi katıldı',
                    inline: false
                })
                .setColor(0x00ff00)
                .setFooter({ 
                    text: `RoveBot\'s • Çekiliş ID: ${giveawayId}`,
                    iconURL: interaction.guild.iconURL() || undefined
                })
                .setTimestamp();

            if (interaction.guild.bannerURL()) {
                giveawayEmbed.setImage(interaction.guild.bannerURL({ size: 1024 }));
            }

            if (interaction.guild.iconURL()) {
                giveawayEmbed.setThumbnail(interaction.guild.iconURL({ size: 256 }));
            }

            const joinButton = new ButtonBuilder()
                .setCustomId(`giveaway_join_${giveawayId}`)
                .setLabel('🎉 Çekilişe Katıl')
                .setStyle(ButtonStyle.Primary);

            const row = new ActionRowBuilder()
                .addComponents(joinButton);

            const giveawayMessage = await channel.send({
                content: '@everyone',
                embeds: [giveawayEmbed],
                components: [row]
            });

            const giveaways = loadGiveaways();
            giveaways[giveawayId] = {
                id: giveawayId,
                title: title,
                description: description,
                prize: prize,
                winnerCount: winnerCount,
                endDate: endDate.toISOString(),
                channelId: channel.id,
                messageId: giveawayMessage.id,
                guildId: interaction.guild.id,
                creatorId: interaction.user.id,
                participants: [],
                status: 'active',
                createdAt: new Date().toISOString()
            };
            saveGiveaways(giveaways);

            const successEmbed = new EmbedBuilder()
                .setColor(0x00ff00)
                .setTitle('✅ Çekiliş Oluşturuldu')
                .setDescription(`Çekiliş başarıyla oluşturuldu!\n\n**Çekiliş ID:** \`${giveawayId}\`\n**Kanal:** <#${channel.id}>\n**Bitiş Tarihi:** <t:${Math.floor(endDate.getTime() / 1000)}:F>`)
                .addFields(
                    {
                        name: '🎁 Ödül',
                        value: prize,
                        inline: true
                    },
                    {
                        name: '👥 Kazanan Sayısı',
                        value: winnerCount.toString(),
                        inline: true
                    },
                    {
                        name: '⏰ Süre',
                        value: `${days} gün`,
                        inline: true
                    }
                )
                .setFooter({ text: `RoveBot\'s • Çekiliş Sistemi` })
                .setTimestamp();

            await interaction.editReply({ embeds: [successEmbed] });

            try {
                const logSistemi = new LogSistemi(interaction.client);
                await logSistemi.logGiveawayOperation({
                    type: 'GIVEAWAY_CREATE',
                    giveawayId: giveawayId,
                    title: title,
                    executor: interaction.user,
                    channel: channel,
                    prize: prize,
                    winnerCount: winnerCount,
                    endDate: endDate,
                    guild: interaction.guild
                });
            } catch (logError) {
                console.error('Çekiliş oluşturma işlemi loglanamadı:', logError);
            }

        } catch (error) {
            console.error('Çekiliş komutu hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription('Çekiliş oluşturulurken bir hata oluştu!')
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    }
};