const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits, StringSelectMenuBuilder, ActionRowBuilder, ComponentType } = require('discord.js');
const fs = require('fs');
const path = require('path');
const LogSistemi = require('../logSistemi');

function checkPermission(interaction, requiredPermission) {
    // Null/undefined kontrolleri ekle
    if (!interaction || !interaction.guild || !interaction.member) {
        console.error('cekilis-sonuclandir.js checkPermission: interaction, guild veya member undefined');
        return false;
    }

    const yetkiPath = path.join(__dirname, '../yetkiSistemi.json');
    
    if (!fs.existsSync(yetkiPath)) {
        return interaction.member.permissions.has(PermissionFlagsBits.Administrator);
    }

    try {
        const yetkiData = JSON.parse(fs.readFileSync(yetkiPath, 'utf8'));
        const guildData = yetkiData[interaction.guild.id];

        if (!guildData || !guildData[requiredPermission]) {
            return interaction.member.permissions.has(PermissionFlagsBits.Administrator);
        }

        return interaction.member.roles.cache.has(guildData[requiredPermission]) || 
               interaction.member.permissions.has(PermissionFlagsBits.Administrator);
    } catch (error) {
        console.error('cekilis-sonuclandir.js checkPermission JSON parse hatası:', error);
        return interaction.member.permissions.has(PermissionFlagsBits.Administrator);
    }
}

function loadGiveaways() {
    const giveawayPath = path.join(__dirname, '../cekilisler.json');
    if (!fs.existsSync(giveawayPath)) {
        return {};
    }
    return JSON.parse(fs.readFileSync(giveawayPath, 'utf8'));
}

function saveGiveaways(data) {
    const giveawayPath = path.join(__dirname, '../cekilisler.json');
    fs.writeFileSync(giveawayPath, JSON.stringify(data, null, 2));
}

function selectWinners(participants, winnerCount) {
    if (participants.length <= winnerCount) {
        return participants;
    }
    
    const shuffled = [...participants].sort(() => 0.5 - Math.random());
    return shuffled.slice(0, winnerCount);
}

module.exports = {
    data: new SlashCommandBuilder()
        .setName('cekilis-sonuclandir')
        .setDescription('Aktif bir çekilişi sonuçlandırır')
        .addStringOption(option =>
            option.setName('cekilisid')
                .setDescription('Sonuçlandırılacak çekiliş ID\'si (boş bırakılırsa menüden seçim yapılır)')
                .setRequired(false)),

    async execute(interaction) {
        try {
            if (!checkPermission(interaction, 'cekilissorumlusu')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetkisiz Erişim')
                    .setDescription('Bu komutu kullanmak için Çekiliş Sorumlusu yetkisine sahip olmalısınız!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply({ ephemeral: true });

            const giveaways = loadGiveaways();
            const guildGiveaways = Object.values(giveaways).filter(g => g.guildId === interaction.guild.id && g.status === 'active');

            if (guildGiveaways.length === 0) {
                const noGiveawaysEmbed = new EmbedBuilder()
                    .setColor(0xffaa00)
                    .setTitle('⚠️ Aktif Çekiliş Bulunamadı')
                    .setDescription('Bu sunucuda aktif çekiliş bulunmuyor!')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [noGiveawaysEmbed] });
            }

            let selectedGiveawayId = interaction.options.getString('cekilisid');

            if (!selectedGiveawayId) {
                if (guildGiveaways.length === 1) {
                    selectedGiveawayId = guildGiveaways[0].id;
                } else {
                    const selectMenu = new StringSelectMenuBuilder()
                        .setCustomId('giveaway_select')
                        .setPlaceholder('Sonuçlandırılacak çekilişi seçin...')
                        .addOptions(
                            guildGiveaways.slice(0, 25).map(giveaway => ({
                                label: giveaway.title.length > 100 ? giveaway.title.substring(0, 97) + '...' : giveaway.title,
                                description: `ID: ${giveaway.id} | Ödül: ${giveaway.prize.length > 50 ? giveaway.prize.substring(0, 47) + '...' : giveaway.prize}`,
                                value: giveaway.id
                            }))
                        );

                    const row = new ActionRowBuilder().addComponents(selectMenu);

                    const selectEmbed = new EmbedBuilder()
                        .setColor(0x0099ff)
                        .setTitle('🎯 Çekiliş Seçimi')
                        .setDescription(`**${guildGiveaways.length}** aktif çekiliş bulundu. Sonuçlandırmak istediğiniz çekilişi seçin:`)
                        .setTimestamp();

                    const response = await interaction.editReply({
                        embeds: [selectEmbed],
                        components: [row]
                    });

                    try {
                        const selectInteraction = await response.awaitMessageComponent({
                            componentType: ComponentType.StringSelect,
                            time: 60000,
                            filter: i => i.user.id === interaction.user.id
                        });

                        selectedGiveawayId = selectInteraction.values[0];
                        await selectInteraction.deferUpdate();
                    } catch (error) {
                        const timeoutEmbed = new EmbedBuilder()
                            .setColor(0xff0000)
                            .setTitle('⏰ Zaman Aşımı')
                            .setDescription('Çekiliş seçimi zaman aşımına uğradı!')
                            .setTimestamp();

                        return await interaction.editReply({ embeds: [timeoutEmbed], components: [] });
                    }
                }
            }

            const selectedGiveaway = giveaways[selectedGiveawayId];
            if (!selectedGiveaway) {
                const notFoundEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Çekiliş Bulunamadı')
                    .setDescription('Belirtilen ID\'ye sahip çekiliş bulunamadı!')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [notFoundEmbed], components: [] });
            }

            if (selectedGiveaway.status !== 'active') {
                const alreadyEndedEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Çekiliş Zaten Sonuçlandı')
                    .setDescription('Bu çekiliş zaten sonuçlandırılmış!')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [alreadyEndedEmbed], components: [] });
            }

            if (selectedGiveaway.participants.length === 0) {
                const noParticipantsEmbed = new EmbedBuilder()
                    .setColor(0xffaa00)
                    .setTitle('⚠️ Katılımcı Bulunamadı')
                    .setDescription('Bu çekilişe hiç katılımcı yok!')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [noParticipantsEmbed], components: [] });
            }

            const winners = selectWinners(selectedGiveaway.participants, selectedGiveaway.winnerCount);

            selectedGiveaway.status = 'ended';
            selectedGiveaway.winners = winners;
            selectedGiveaway.endedAt = new Date().toISOString();
            selectedGiveaway.endedBy = interaction.user.id;
            giveaways[selectedGiveawayId] = selectedGiveaway;
            saveGiveaways(giveaways);

            try {
                const channel = interaction.guild.channels.cache.get(selectedGiveaway.channelId);
                if (channel) {
                    const message = await channel.messages.fetch(selectedGiveaway.messageId);
                    if (message) {
                        const updatedEmbed = EmbedBuilder.from(message.embeds[0])
                            .setDescription(`${selectedGiveaway.description}\n\n🎁 **Ödül:** ${selectedGiveaway.prize}\n👥 **Kazanan Sayısı:** ${selectedGiveaway.winnerCount}\n⏰ **Bitiş Tarihi:** <t:${Math.floor(new Date(selectedGiveaway.endDate).getTime() / 1000)}:F>\n🔴 **Durum:** Sonuçlandı\n👤 **Oluşturan:** <@${selectedGiveaway.creatorId}>\n\n🏆 **Kazananlar:** ${winners.map(w => `<@${w}>`).join(', ')}`)
                            .setColor(0xff0000);

                        updatedEmbed.addFields({
                            name: '👥 Katılımcılar',
                            value: `**${selectedGiveaway.participants.length}** kişi katıldı (Çekiliş sona erdi)`,
                            inline: false
                        });

                        await message.edit({ embeds: [updatedEmbed], components: [] });
                    }
                }
            } catch (error) {
                console.error('Çekiliş mesajı güncellenemedi:', error);
            }

            const resultEmbed = new EmbedBuilder()
                .setColor(0x00ff00)
                .setTitle('🎉 Çekiliş Sonuçlandı!')
                .setDescription(`**${selectedGiveaway.title}** çekilişi sonuçlandı!`)
                .addFields(
                    {
                        name: '🎁 Ödül',
                        value: selectedGiveaway.prize,
                        inline: true
                    },
                    {
                        name: '👥 Toplam Katılımcı',
                        value: selectedGiveaway.participants.length.toString(),
                        inline: true
                    },
                    {
                        name: '🏆 Kazanan Sayısı',
                        value: winners.length.toString(),
                        inline: true
                    },
                    {
                        name: '🏆 Kazananlar',
                        value: winners.length > 0 ? winners.map(w => `<@${w}>`).join('\n') : 'Kazanan yok',
                        inline: false
                    }
                )
                .setFooter({ text: `RoveBot\'s • Çekiliş ID: ${selectedGiveawayId}` })
                .setTimestamp();

            try {
                const channel = interaction.guild.channels.cache.get(selectedGiveaway.channelId);
                if (channel) {
                    await channel.send({
                        content: winners.length > 0 ? winners.map(w => `<@${w}>`).join(' ') : '',
                        embeds: [resultEmbed]
                    });
                }
            } catch (error) {
                console.error('Sonuç mesajı gönderilemedi:', error);
            }

            const successEmbed = new EmbedBuilder()
                .setColor(0x00ff00)
                .setTitle('✅ Çekiliş Sonuçlandırıldı')
                .setDescription(`**${selectedGiveaway.title}** çekilişi başarıyla sonuçlandırıldı!\n\n**Kazananlar:** ${winners.length > 0 ? winners.map(w => `<@${w}>`).join(', ') : 'Kazanan yok'}`)
                .setTimestamp();

            await interaction.editReply({ embeds: [successEmbed], components: [] });

            try {
                const logSistemi = new LogSistemi(interaction.client);
                await logSistemi.logGiveawayOperation({
                    type: 'GIVEAWAY_END',
                    giveawayId: selectedGiveawayId,
                    title: selectedGiveaway.title,
                    executor: interaction.user,
                    winners: winners,
                    participantCount: selectedGiveaway.participants.length,
                    guild: interaction.guild
                });
            } catch (logError) {
                console.error('Çekiliş sonuçlandırma işlemi loglanamadı:', logError);
            }

        } catch (error) {
            console.error('Çekiliş sonuçlandırma komutu hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription('Çekiliş sonuçlandırılırken bir hata oluştu!')
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed], components: [] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    }
};