const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits } = require('discord.js');
const fs = require('fs');
const path = require('path');
const LogSistemi = require('../logSistemi');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('banaffı')
        .setDescription('Sunucudaki tüm banları kaldırır (Sadece Owner)'),

    async execute(interaction) {
        try {
            await interaction.deferReply();

            const configPath = path.join(__dirname, '../../config.json');
            const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));

            const ownerRoleId = config.LogSistemi?.logRolleri?.ownerRolu;
            if (!ownerRoleId) {
                const errorEmbed = new EmbedBuilder()
                    .setTitle('❌ Yapılandırma Hatası')
                    .setDescription('Owner rolü tanımlanmamış! Lütfen `/setuplog` komutunu kullanarak log sistemini kurun.')
                    .setColor('#ff0000')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            const member = interaction.member;
            if (!member.roles.cache.has(ownerRoleId)) {
                const noPermEmbed = new EmbedBuilder()
                    .setTitle('🚫 Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için Owner rolüne sahip olmalısınız!')
                    .setColor('#ff0000')
                    .setTimestamp()
                    .addFields({
                        name: '🔒 Gerekli Rol',
                        value: `<@&${ownerRoleId}>`,
                        inline: false
                    });

                return await interaction.editReply({ embeds: [noPermEmbed] });
            }

            const bans = await interaction.guild.bans.fetch();
            
            if (bans.size === 0) {
                const noBansEmbed = new EmbedBuilder()
                    .setTitle('ℹ️ Ban Bulunamadı')
                    .setDescription('Sunucuda hiç ban bulunmuyor.')
                    .setColor('#0099ff')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [noBansEmbed] });
            }

            const startEmbed = new EmbedBuilder()
                .setTitle('🔄 Ban Affı İşlemi Başlatılıyor...')
                .setDescription(`**${bans.size}** adet ban bulundu. Tüm banlar kaldırılıyor...`)
                .setColor('#ffaa00')
                .setTimestamp()
                .addFields({
                    name: '👤 İşlemi Başlatan',
                    value: `${interaction.user}`,
                    inline: true
                },
                {
                    name: '📊 Toplam Ban',
                    value: `${bans.size}`,
                    inline: true
                });

            await interaction.editReply({ embeds: [startEmbed] });

            const logSistemi = new LogSistemi(interaction.client);
            let successCount = 0;
            let failCount = 0;
            const failedUsers = [];

            for (const [userId, banInfo] of bans) {
                try {
                    await interaction.guild.members.unban(userId, `Ban affı - ${interaction.user.tag} tarafından toplu ban affı`);
                    successCount++;

                    await logSistemi.logBanAffi(interaction.guild, {
                        unbannedUser: banInfo.user,
                        moderator: interaction.user,
                        reason: banInfo.reason || 'Sebep belirtilmemiş',
                        originalBanReason: banInfo.reason || 'Bilinmiyor'
                    });

                    await new Promise(resolve => setTimeout(resolve, 100));

                } catch (error) {
                    console.error(`${banInfo.user.tag} ban affı hatası:`, error);
                    failCount++;
                    failedUsers.push({
                        user: banInfo.user,
                        error: error.message
                    });
                }
            }

            const resultEmbed = new EmbedBuilder()
                .setTitle('✅ Ban Affı İşlemi Tamamlandı')
                .setDescription(`Toplu ban affı işlemi tamamlandı!`)
                .setColor(failCount > 0 ? '#ffaa00' : '#00ff00')
                .setTimestamp()
                .addFields(
                    {
                        name: '✅ Başarılı',
                        value: `${successCount} ban kaldırıldı`,
                        inline: true
                    },
                    {
                        name: '❌ Başarısız',
                        value: `${failCount} ban kaldırılamadı`,
                        inline: true
                    },
                    {
                        name: '👤 İşlemi Yapan',
                        value: `${interaction.user}`,
                        inline: true
                    }
                );

            if (failedUsers.length > 0 && failedUsers.length <= 10) {
                const failedList = failedUsers.map(f => `• ${f.user.tag} - ${f.error.substring(0, 50)}...`).join('\n');
                resultEmbed.addFields({
                    name: '⚠️ Başarısız Banlar',
                    value: failedList,
                    inline: false
                });
            } else if (failedUsers.length > 10) {
                resultEmbed.addFields({
                    name: '⚠️ Başarısız Banlar',
                    value: `${failedUsers.length} ban kaldırılamadı. Detaylar için loglara bakın.`,
                    inline: false
                });
            }

            await interaction.editReply({ embeds: [resultEmbed] });

            await logSistemi.logMassBanAffi(interaction.guild, {
                moderator: interaction.user,
                totalBans: bans.size,
                successCount: successCount,
                failCount: failCount,
                failedUsers: failedUsers
            });

        } catch (error) {
            console.error('Ban affı komutu hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setTitle('❌ Hata Oluştu')
                .setDescription(`Ban affı işlemi sırasında bir hata oluştu:\n\`\`\`${error.message}\`\`\``)
                .setColor('#ff0000')
                .setTimestamp();

            if (interaction.replied || interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    }
};