const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits } = require('discord.js');
const fs = require('fs');
const path = require('path');
const LogSistemi = require('../logSistemi');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('ban')
        .setDescription('Ban yönetim sistemi - Kullanıcı banla, ban aç veya ban bilgisi görüntüle')
        .addStringOption(option =>
            option.setName('islem')
                .setDescription('Yapılacak işlem türü')
                .setRequired(true)
                .addChoices(
                    { name: 'Kullanıcı Banla', value: 'banla' },
                    { name: 'Kullanıcının Banını Aç', value: 'unban' },
                    { name: 'Kullanıcı Ban Bilgisi', value: 'bilgi' }
                ))
        .addUserOption(option =>
            option.setName('kullanici')
                .setDescription('İşlem yapılacak kullanıcı')
                .setRequired(true))
        .addStringOption(option =>
            option.setName('sebep')
                .setDescription('İşlem sebebi')
                .setRequired(false))
        .addIntegerOption(option =>
            option.setName('mesaj_sil')
                .setDescription('Kaç günlük mesajları silinsin? (0-7 gün, sadece ban işlemi için)')
                .setMinValue(0)
                .setMaxValue(7)
                .setRequired(false)),

    async execute(interaction) {
        try {
            const islem = interaction.options.getString('islem');
            const targetUser = interaction.options.getUser('kullanici');
            const reason = interaction.options.getString('sebep') || 'Sebep belirtilmedi';
            const deleteMessageDays = interaction.options.getInteger('mesaj_sil') || 0;

            switch (islem) {
                case 'banla':
                    return await this.handleBan(interaction, targetUser, reason, deleteMessageDays);
                case 'unban':
                    return await this.handleUnban(interaction, targetUser, reason);
                case 'bilgi':
                    return await this.handleBanInfo(interaction, targetUser);
                default:
                    return await interaction.reply({ 
                        content: '❌ Geçersiz işlem türü!', 
                        ephemeral: true 
                    });
            }
        } catch (error) {
            console.error('Ban komutu genel hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription('Komut işlenirken beklenmeyen bir hata oluştu!')
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    },

    async handleBan(interaction, targetUser, reason, deleteMessageDays) {
        try {
            const hasPermission = await this.checkPermission(interaction, 'ban');
            if (!hasPermission) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için yeterli yetkiniz yok!')
                    .addFields({
                        name: '📋 Gerekli Yetkiler',
                        value: '• Ban Members yetkisi\n• Yetkili rol (eğer ayarlanmışsa)\n• Administrator yetkisi'
                    })
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            if (targetUser.id === interaction.user.id) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription('Kendinizi yasaklayamazsınız!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            if (targetUser.id === interaction.client.user.id) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription('Beni yasaklayamazsınız!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            try {
                const banInfo = await interaction.guild.bans.fetch(targetUser.id);
                if (banInfo) {
                    const errorEmbed = new EmbedBuilder()
                        .setColor(0xff0000)
                        .setTitle('❌ Kullanıcı Zaten Banlı')
                        .setDescription(`**${targetUser.tag}** zaten sunucudan yasaklı!`)
                        .addFields({
                            name: '📝 Mevcut Ban Sebebi',
                            value: banInfo.reason || 'Sebep belirtilmemiş',
                            inline: false
                        })
                        .setTimestamp();

                    return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
                }
            } catch (error) {
            }

            let targetMember = null;
            try {
                targetMember = await interaction.guild.members.fetch(targetUser.id);
            } catch (error) {
            }

            if (targetMember) {
                const botMember = interaction.guild.members.me;
                if (targetMember.roles.highest.position >= botMember.roles.highest.position) {
                    const errorEmbed = new EmbedBuilder()
                        .setColor(0xff0000)
                        .setTitle('❌ Yetki Hatası')
                        .setDescription('Bu kullanıcıyı yasaklayamam! Kullanıcının rolü benim rolümden yüksek veya eşit.')
                        .setTimestamp();

                    return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
                }

                if (targetMember.roles.highest.position >= interaction.member.roles.highest.position && 
                    !interaction.member.permissions.has(PermissionFlagsBits.Administrator)) {
                    const errorEmbed = new EmbedBuilder()
                        .setColor(0xff0000)
                        .setTitle('❌ Yetki Hatası')
                        .setDescription('Bu kullanıcıyı yasaklayamazsınız! Kullanıcının rolü sizin rolünüzden yüksek veya eşit.')
                        .setTimestamp();

                    return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
                }

                if (targetMember.id === interaction.guild.ownerId) {
                    const errorEmbed = new EmbedBuilder()
                        .setColor(0xff0000)
                        .setTitle('❌ Hata')
                        .setDescription('Sunucu sahibini yasaklayamazsınız!')
                        .setTimestamp();

                    return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
                }
            }

            await interaction.deferReply();

            try {
                const dmEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('🚫 Sunucudan Yasaklandınız')
                    .setDescription(`**${interaction.guild.name}** sunucusundan yasaklandınız.`)
                    .addFields(
                        {
                            name: '👮 Yasaklayan Yetkili',
                            value: `${interaction.user.tag}`,
                            inline: true
                        },
                        {
                            name: '📝 Sebep',
                            value: reason,
                            inline: true
                        },
                        {
                            name: '🕐 Tarih',
                            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                            inline: false
                        }
                    )
                    .setThumbnail(interaction.guild.iconURL({ dynamic: true }))
                    .setTimestamp();

                await targetUser.send({ embeds: [dmEmbed] });
            } catch (error) {
            }

            await interaction.guild.members.ban(targetUser.id, {
                deleteMessageSeconds: deleteMessageDays * 24 * 60 * 60,
                reason: `${reason} | Yasaklayan: ${interaction.user.tag}`
            });

            await this.logUnbanAction(interaction, targetUser, 'ban', reason, interaction.user, deleteMessageDays);

            const successEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('🚫 Kullanıcı Yasaklandı')
                .setDescription(`**${targetUser.tag}** başarıyla sunucudan yasaklandı.`)
                .addFields(
                    {
                        name: '👤 Yasaklanan Kullanıcı',
                        value: `${targetUser.tag} (${targetUser.id})`,
                        inline: true
                    },
                    {
                        name: '👮 Yasaklayan Yetkili',
                        value: `${interaction.user.tag}`,
                        inline: true
                    },
                    {
                        name: '📝 Sebep',
                        value: reason,
                        inline: false
                    }
                )
                .setThumbnail(targetUser.displayAvatarURL({ dynamic: true }))
                .setTimestamp();

            if (deleteMessageDays > 0) {
                successEmbed.addFields({
                    name: '🗑️ Silinen Mesajlar',
                    value: `Son ${deleteMessageDays} günlük mesajlar silindi`,
                    inline: true
                });
            }

            await interaction.editReply({ embeds: [successEmbed] });

        } catch (error) {
            console.error('Ban işlemi hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Ban Hatası')
                .setDescription('Kullanıcı yasaklanırken bir hata oluştu!')
                .addFields({
                    name: '🔍 Olası Sebepler',
                    value: '• Kullanıcı zaten yasaklı olabilir\n• Bot yetkisi yetersiz olabilir\n• Bilinmeyen bir hata oluştu'
                })
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    },

    async handleUnban(interaction, targetUser, reason) {
        try {
            const hasPermission = await this.checkPermission(interaction, 'ban');
            if (!hasPermission) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için yeterli yetkiniz yok!')
                    .addFields({
                        name: '📋 Gerekli Yetkiler',
                        value: '• Ban Members yetkisi\n• Yetkili rol (eğer ayarlanmışsa)\n• Administrator yetkisi'
                    })
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            let banInfo = null;
            try {
                banInfo = await interaction.guild.bans.fetch(targetUser.id);
            } catch (error) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Kullanıcı Banlı Değil')
                    .setDescription(`**${targetUser.tag}** sunucudan yasaklı değil!`)
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            await interaction.guild.members.unban(targetUser.id, `${reason} | Affeden: ${interaction.user.tag}`);

            try {
                const dmEmbed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('✅ Ban Affı')
                    .setDescription(`**${interaction.guild.name}** sunucusundaki banınız kaldırıldı.`)
                    .addFields(
                        {
                            name: '👮 Affeden Yetkili',
                            value: `${interaction.user.tag}`,
                            inline: true
                        },
                        {
                            name: '📝 Sebep',
                            value: reason,
                            inline: true
                        },
                        {
                            name: '🕐 Tarih',
                            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                            inline: false
                        },
                        {
                            name: '🔗 Davet Linki',
                            value: 'Sunucuya tekrar katılabilirsiniz!',
                            inline: false
                        }
                    )
                    .setThumbnail(interaction.guild.iconURL({ dynamic: true }))
                    .setTimestamp();

                await targetUser.send({ embeds: [dmEmbed] });
            } catch (error) {
            }

            await this.logUnbanAction(interaction, targetUser, 'unban', reason, interaction.user);

            const successEmbed = new EmbedBuilder()
                .setColor(0x00ff00)
                .setTitle('✅ Ban Affı Başarılı')
                .setDescription(`**${targetUser.tag}** kullanıcısının banı başarıyla kaldırıldı.`)
                .addFields(
                    {
                        name: '👤 Affedilen Kullanıcı',
                        value: `${targetUser.tag} (${targetUser.id})`,
                        inline: true
                    },
                    {
                        name: '👮 Affeden Yetkili',
                        value: `${interaction.user.tag}`,
                        inline: true
                    },
                    {
                        name: '📝 Sebep',
                        value: reason,
                        inline: false
                    },
                    {
                        name: '📝 Eski Ban Sebebi',
                        value: banInfo.reason || 'Sebep belirtilmemiş',
                        inline: false
                    }
                )
                .setThumbnail(targetUser.displayAvatarURL({ dynamic: true }))
                .setTimestamp();

            await interaction.editReply({ embeds: [successEmbed] });

        } catch (error) {
            console.error('Unban işlemi hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Unban Hatası')
                .setDescription('Ban kaldırılırken bir hata oluştu!')
                .addFields({
                    name: '🔍 Olası Sebepler',
                    value: '• Kullanıcı zaten banlı değil\n• Bot yetkisi yetersiz olabilir\n• Bilinmeyen bir hata oluştu'
                })
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    },

    async handleBanInfo(interaction, targetUser) {
        try {
            await interaction.deferReply();

            let banInfo = null;
            try {
                banInfo = await interaction.guild.bans.fetch(targetUser.id);
            } catch (error) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('✅ Kullanıcı Banlı Değil')
                    .setDescription(`**${targetUser.tag}** sunucudan yasaklı değil.`)
                    .addFields({
                        name: '👤 Kullanıcı Bilgileri',
                        value: `**Tag:** ${targetUser.tag}\n**ID:** ${targetUser.id}\n**Hesap Oluşturma:** <t:${Math.floor(targetUser.createdTimestamp / 1000)}:F>`,
                        inline: false
                    })
                    .setThumbnail(targetUser.displayAvatarURL({ dynamic: true }))
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            const banInfoEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('🚫 Kullanıcı Ban Bilgileri')
                .setDescription(`**${targetUser.tag}** sunucudan yasaklı.`)
                .addFields(
                    {
                        name: '👤 Yasaklı Kullanıcı',
                        value: `**Tag:** ${targetUser.tag}\n**ID:** ${targetUser.id}\n**Hesap Oluşturma:** <t:${Math.floor(targetUser.createdTimestamp / 1000)}:F>`,
                        inline: false
                    },
                    {
                        name: '📝 Ban Sebebi',
                        value: banInfo.reason || 'Sebep belirtilmemiş',
                        inline: false
                    }
                )
                .setThumbnail(targetUser.displayAvatarURL({ dynamic: true }))
                .setTimestamp();

            await interaction.editReply({ embeds: [banInfoEmbed] });

        } catch (error) {
            console.error('Ban bilgisi hatası:', error);
            
            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Bilgi Hatası')
                .setDescription('Ban bilgileri alınırken bir hata oluştu!')
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }
        }
    },

    async logUnbanAction(interaction, targetUser, action, reason, executor, deleteMessageDays = null) {
        try {
            const logSistemi = new LogSistemi(interaction.client);
            await logSistemi.logUnbanOperation(targetUser, action, reason, executor, deleteMessageDays);
        } catch (error) {
            console.error('Unban log hatası:', error);
        }
    },

    async checkPermission(interaction, action) {
        // Null/undefined kontrolleri ekle
        if (!interaction || !interaction.guild || !interaction.member) {
            console.error('ban.js checkPermission: interaction, guild veya member undefined');
            return false;
        }

        if (interaction.member.permissions.has(PermissionFlagsBits.Administrator)) {
            return true;
        }

        if (action === 'ban' && !interaction.member.permissions.has(PermissionFlagsBits.BanMembers)) {
            return false;
        }

        const yetkiPath = path.join(__dirname, '../yetkiSistemi.json');
        if (fs.existsSync(yetkiPath)) {
            try {
                const yetkiData = JSON.parse(fs.readFileSync(yetkiPath, 'utf8'));
                const guildData = yetkiData[interaction.guild.id];
                
                if (guildData && guildData[action]) {
                    return interaction.member.roles.cache.has(guildData[action]);
                }
            } catch (error) {
                console.error('ban.js checkPermission JSON parse hatası:', error);
                return false;
            }
        }

        return false;
    }
};