const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits, AttachmentBuilder } = require('discord.js');
const fs = require('fs');
const path = require('path');
const LogSistemi = require('../logSistemi');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('backup')
        .setDescription('Sunucunun tam yapısal yedeğini alır (Sadece owner rolü)'),

    async execute(interaction) {
        try {
            await interaction.deferReply({ ephemeral: true });

            const configPath = path.join(__dirname, '../../config.json');
            const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));

            const ownerRoleId = config.LogSistemi?.logRolleri?.ownerRolu;
            if (!ownerRoleId) {
                const errorEmbed = new EmbedBuilder()
                    .setTitle('❌ Yapılandırma Hatası')
                    .setDescription('Owner rolü tanımlanmamış! Lütfen `/setuplog` komutunu kullanarak log sistemini kurun.')
                    .setColor('#ff0000')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            const logSistemi = new LogSistemi(interaction.client);
            await logSistemi.logBackupStart(interaction.guild.id, interaction.user.id);

            const progressEmbed = new EmbedBuilder()
                .setTitle('💾 Yedekleme İşlemi Başlatılıyor...')
                .setDescription('Sunucu yapısı analiz ediliyor ve yedek oluşturuluyor...')
                .setColor('#ffaa00')
                .setTimestamp();

            await interaction.editReply({ embeds: [progressEmbed] });

            const backupData = {
                metadata: {
                    version: '1.0',
                    timestamp: new Date().toISOString(),
                    serverId: interaction.guild.id,
                    serverName: interaction.guild.name,
                    createdBy: interaction.user.id,
                    memberCount: interaction.guild.memberCount
                },
                guild: {
                    name: interaction.guild.name,
                    description: interaction.guild.description,
                    icon: interaction.guild.iconURL(),
                    banner: interaction.guild.bannerURL(),
                    verificationLevel: interaction.guild.verificationLevel,
                    defaultMessageNotifications: interaction.guild.defaultMessageNotifications,
                    explicitContentFilter: interaction.guild.explicitContentFilter,
                    afkChannelId: interaction.guild.afkChannelId,
                    afkTimeout: interaction.guild.afkTimeout,
                    systemChannelId: interaction.guild.systemChannelId,
                    rulesChannelId: interaction.guild.rulesChannelId,
                    publicUpdatesChannelId: interaction.guild.publicUpdatesChannelId
                },
                channels: [],
                roles: [],
                userRoles: {},
                emojis: []
            };

            const channels = interaction.guild.channels.cache.filter(channel => channel.type !== 4);
            const categories = interaction.guild.channels.cache.filter(channel => channel.type === 4);

            categories.forEach(category => {
                backupData.channels.push({
                    id: category.id,
                    name: category.name,
                    type: category.type,
                    position: category.position,
                    parentId: null,
                    permissionOverwrites: category.permissionOverwrites?.cache ? 
                        category.permissionOverwrites.cache.map(overwrite => ({
                            id: overwrite.id,
                            type: overwrite.type,
                            allow: overwrite.allow.bitfield.toString(),
                            deny: overwrite.deny.bitfield.toString()
                        })) : []
                });
            });

            channels.forEach(channel => {
                const channelData = {
                    id: channel.id,
                    name: channel.name,
                    type: channel.type,
                    position: channel.position,
                    parentId: channel.parentId,
                    permissionOverwrites: channel.permissionOverwrites?.cache ? 
                        channel.permissionOverwrites.cache.map(overwrite => ({
                            id: overwrite.id,
                            type: overwrite.type,
                            allow: overwrite.allow.bitfield.toString(),
                            deny: overwrite.deny.bitfield.toString()
                        })) : []
                };

                if (channel.isTextBased()) {
                    channelData.topic = channel.topic;
                    channelData.nsfw = channel.nsfw;
                    channelData.rateLimitPerUser = channel.rateLimitPerUser;
                }

                if (channel.isVoiceBased()) {
                    channelData.bitrate = channel.bitrate;
                    channelData.userLimit = channel.userLimit;
                    channelData.rtcRegion = channel.rtcRegion;
                }

                backupData.channels.push(channelData);
            });

            interaction.guild.roles.cache.forEach(role => {
                if (role.id !== interaction.guild.id) {
                    backupData.roles.push({
                        id: role.id,
                        name: role.name,
                        color: role.color,
                        hoist: role.hoist,
                        position: role.position,
                        permissions: role.permissions.bitfield.toString(),
                        mentionable: role.mentionable,
                        managed: role.managed,
                        icon: role.iconURL(),
                        unicodeEmoji: role.unicodeEmoji
                    });
                }
            });

            interaction.guild.members.cache.forEach(member => {
                const userRoles = member.roles.cache
                    .filter(role => role.id !== interaction.guild.id)
                    .map(role => role.id);
                
                if (userRoles.length > 0) {
                    backupData.userRoles[member.id] = userRoles;
                }
            });

            interaction.guild.emojis.cache.forEach(emoji => {
                backupData.emojis.push({
                    id: emoji.id,
                    name: emoji.name,
                    animated: emoji.animated,
                    url: emoji.url,
                    managed: emoji.managed,
                    available: emoji.available,
                    roles: emoji.roles.cache.map(role => role.id)
                });
            });

            const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
            const fileName = `backup_${interaction.guild.id}_${timestamp}.json`;
            const backupDir = path.join(__dirname, '../../backups');
            
            if (!fs.existsSync(backupDir)) {
                fs.mkdirSync(backupDir, { recursive: true });
            }

            const filePath = path.join(backupDir, fileName);
            fs.writeFileSync(filePath, JSON.stringify(backupData, null, 2), 'utf8');

            const stats = {
                channels: backupData.channels.length,
                roles: backupData.roles.length,
                userRoles: Object.keys(backupData.userRoles).length,
                emojis: backupData.emojis.length
            };

            const successEmbed = new EmbedBuilder()
                .setTitle('✅ Yedekleme Tamamlandı!')
                .setDescription(`Sunucu yapısı başarıyla yedeklendi.`)
                .addFields(
                    { name: '📁 Dosya Adı', value: fileName, inline: false },
                    { name: '📊 İstatistikler', value: `**Kanallar:** ${stats.channels}\n**Roller:** ${stats.roles}\n**Kullanıcı Rolleri:** ${stats.userRoles}\n**Emojiler:** ${stats.emojis}`, inline: false },
                    { name: '📅 Oluşturulma Tarihi', value: new Date().toLocaleString('tr-TR', { timeZone: 'Europe/Istanbul' }), inline: false }
                )
                .setColor('#00ff00')
                .setTimestamp();

            const attachment = new AttachmentBuilder(filePath, { name: fileName });

            await interaction.editReply({ 
                embeds: [successEmbed],
                files: [attachment]
            });

            await logSistemi.logBackupSuccess(interaction.guild.id, interaction.user.id, fileName, stats);

        } catch (error) {
            console.error('Backup komutu hatası:', error);

            const logSistemi = new LogSistemi(interaction.client);
            await logSistemi.logBackupError(interaction.guild.id, interaction.user.id, error.message);

            const errorEmbed = new EmbedBuilder()
                .setTitle('❌ Yedekleme Hatası')
                .setDescription(`Yedekleme işlemi sırasında bir hata oluştu:\n\`\`\`${error.message}\`\`\``)
                .setColor('#ff0000')
                .setTimestamp();

            await interaction.editReply({ embeds: [errorEmbed] });
        }
    }
};