const { SlashCommandBuilder, EmbedBuilder, PermissionFlagsBits, AttachmentBuilder, ButtonBuilder, ButtonStyle, ActionRowBuilder } = require('discord.js');
const fs = require('fs');
const path = require('path');
const LogSistemi = require('../logSistemi');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('backup-kur')
        .setDescription('Yedek dosyasından sunucu yapısını geri yükler (Sadece owner rolü)')
        .addAttachmentOption(option =>
            option.setName('yedek-dosyasi')
                .setDescription('Geri yüklenecek JSON yedek dosyası')
                .setRequired(true)),

    async execute(interaction) {
        try {
            await interaction.deferReply({ ephemeral: true });

            const configPath = path.join(__dirname, '../../config.json');
            const config = JSON.parse(fs.readFileSync(configPath, 'utf8'));

            const ownerRoleId = config.LogSistemi?.logRolleri?.ownerRolu;
            if (!ownerRoleId) {
                const errorEmbed = new EmbedBuilder()
                    .setTitle('❌ Yapılandırma Hatası')
                    .setDescription('Owner rolü tanımlanmamış! Lütfen `/setuplog` komutunu kullanarak log sistemini kurun.')
                    .setColor('#ff0000')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            const attachment = interaction.options.getAttachment('yedek-dosyasi');
            
            if (!attachment.name.endsWith('.json')) {
                const errorEmbed = new EmbedBuilder()
                    .setTitle('❌ Geçersiz Dosya')
                    .setDescription('Lütfen geçerli bir JSON yedek dosyası yükleyin!')
                    .setColor('#ff0000')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            const logSistemi = new LogSistemi(interaction.client);

            let backupData;
            try {
                const response = await fetch(attachment.url);
                const jsonText = await response.text();
                backupData = JSON.parse(jsonText);

                if (!backupData.metadata || !backupData.guild || !backupData.channels || !backupData.roles) {
                    await logSistemi.logBackupValidation(interaction.guild.id, interaction.user.id, attachment.name, false, 'Geçersiz yedek dosyası yapısı');
                    
                    const errorEmbed = new EmbedBuilder()
                        .setTitle('❌ Geçersiz Yedek Dosyası')
                        .setDescription('Yedek dosyası geçerli bir yapıya sahip değil!')
                        .setColor('#ff0000')
                        .setTimestamp();

                    return await interaction.editReply({ embeds: [errorEmbed] });
                }

                await logSistemi.logBackupValidation(interaction.guild.id, interaction.user.id, attachment.name, true);

            } catch (error) {
                await logSistemi.logBackupValidation(interaction.guild.id, interaction.user.id, attachment.name, false, error.message);
                
                const errorEmbed = new EmbedBuilder()
                    .setTitle('❌ Dosya Okuma Hatası')
                    .setDescription(`Yedek dosyası okunamadı:\n\`\`\`${error.message}\`\`\``)
                    .setColor('#ff0000')
                    .setTimestamp();

                return await interaction.editReply({ embeds: [errorEmbed] });
            }

            const confirmEmbed = new EmbedBuilder()
                .setTitle('⚠️ Geri Yükleme Onayı')
                .setDescription(`**DİKKAT:** Bu işlem mevcut sunucu yapısını tamamen silecek ve yedekten geri yükleyecektir!\n\n**Yedek Bilgileri:**\n• **Sunucu:** ${backupData.metadata.serverName}\n• **Tarih:** ${new Date(backupData.metadata.timestamp).toLocaleString('tr-TR')}\n• **Kanallar:** ${backupData.channels.length}\n• **Roller:** ${backupData.roles.length}\n• **Kullanıcı Rolleri:** ${Object.keys(backupData.userRoles || {}).length}\n• **Emojiler:** ${backupData.emojis.length}\n\n**Bu işlem geri alınamaz!**`)
                .setColor('#ff6600')
                .setTimestamp();

            const confirmButton = new ButtonBuilder()
                .setCustomId('confirm_restore')
                .setLabel('Geri Yükle')
                .setStyle(ButtonStyle.Danger)
                .setEmoji('✅');

            const cancelButton = new ButtonBuilder()
                .setCustomId('cancel_restore')
                .setLabel('İptal Et')
                .setStyle(ButtonStyle.Secondary)
                .setEmoji('❌');

            const row = new ActionRowBuilder()
                .addComponents(confirmButton, cancelButton);

            const confirmMessage = await interaction.editReply({ 
                embeds: [confirmEmbed], 
                components: [row] 
            });

            const filter = (i) => i.user.id === interaction.user.id;
            const collector = confirmMessage.createMessageComponentCollector({ filter, time: 30000 });

            collector.on('collect', async (i) => {
                if (i.customId === 'cancel_restore') {
                    const cancelEmbed = new EmbedBuilder()
                        .setTitle('❌ İşlem İptal Edildi')
                        .setDescription('Geri yükleme işlemi iptal edildi.')
                        .setColor('#ff0000')
                        .setTimestamp();

                    await i.update({ embeds: [cancelEmbed], components: [] });
                    return;
                }

                if (i.customId === 'confirm_restore') {
                    await i.update({ 
                        embeds: [new EmbedBuilder()
                            .setTitle('🔄 Geri Yükleme Başlatılıyor...')
                            .setDescription('Sunucu yapısı geri yükleniyor, lütfen bekleyin...')
                            .setColor('#ffaa00')
                            .setTimestamp()], 
                        components: [] 
                    });

                    await logSistemi.logRestoreStart(interaction.guild.id, interaction.user.id, attachment.name);

                    try {
                        const stats = await this.performRestore(interaction.guild, backupData, i);
                        
                        const successEmbed = new EmbedBuilder()
                            .setTitle('✅ Geri Yükleme Tamamlandı!')
                            .setDescription('Sunucu yapısı başarıyla geri yüklendi.')
                            .addFields(
                                { name: '📁 Dosya', value: attachment.name, inline: false },
                                { name: '📊 İstatistikler', value: `**Kanallar:** ${stats.channels}\n**Roller:** ${stats.roles}\n**Kullanıcı Rolleri:** ${stats.userRoles}\n**Emojiler:** ${stats.emojis}`, inline: false },
                                { name: '📅 Tamamlanma Tarihi', value: new Date().toLocaleString('tr-TR', { timeZone: 'Europe/Istanbul' }), inline: false }
                            )
                            .setColor('#00ff00')
                            .setTimestamp();

                        await i.editReply({ embeds: [successEmbed] });
                        await logSistemi.logRestoreSuccess(interaction.guild.id, interaction.user.id, attachment.name, stats);

                    } catch (error) {
                        console.error('Geri yükleme hatası:', error);
                        await logSistemi.logRestoreError(interaction.guild.id, interaction.user.id, attachment.name, error.message);

                        const errorEmbed = new EmbedBuilder()
                            .setTitle('❌ Geri Yükleme Hatası')
                            .setDescription(`Geri yükleme sırasında bir hata oluştu:\n\`\`\`${error.message}\`\`\``)
                            .setColor('#ff0000')
                            .setTimestamp();

                        await i.editReply({ embeds: [errorEmbed] });
                    }
                }
            });

            collector.on('end', async (collected) => {
                if (collected.size === 0) {
                    const timeoutEmbed = new EmbedBuilder()
                        .setTitle('⏰ Zaman Aşımı')
                        .setDescription('Onay bekleme süresi doldu. İşlem iptal edildi.')
                        .setColor('#ff6600')
                        .setTimestamp();

                    await interaction.editReply({ embeds: [timeoutEmbed], components: [] });
                }
            });

        } catch (error) {
            console.error('Backup-kur komutu hatası:', error);

            const logSistemi = new LogSistemi(interaction.client);
            await logSistemi.logRestoreError(interaction.guild.id, interaction.user.id, 'Bilinmeyen', error.message);

            const errorEmbed = new EmbedBuilder()
                .setTitle('❌ Sistem Hatası')
                .setDescription(`Bir sistem hatası oluştu:\n\`\`\`${error.message}\`\`\``)
                .setColor('#ff0000')
                .setTimestamp();

            await interaction.editReply({ embeds: [errorEmbed] });
        }
    },

    async performRestore(guild, backupData, interaction) {
        const stats = { channels: 0, roles: 0, userRoles: 0, emojis: 0 };
        await this.clearGuild(guild);
        await this.restoreGuildSettings(guild, backupData.guild);
        const roleMap = await this.restoreRoles(guild, backupData.roles);
        stats.roles = Object.keys(roleMap).length;
        const channelMap = await this.restoreChannels(guild, backupData.channels);
        stats.channels = Object.keys(channelMap).length;
        stats.userRoles = await this.restoreUserRoles(guild, backupData.userRoles, roleMap);
        stats.emojis = await this.restoreEmojis(guild, backupData.emojis);
        return stats;
    },

    async clearGuild(guild) {
        const channels = guild.channels.cache.filter(channel => channel.type !== 4 && channel.deletable);
        for (const channel of channels.values()) {
            try {
                await channel.delete();
            } catch (error) {
                console.error(`Kanal silinemedi: ${channel.name}`, error);
            }
        }

        const categories = guild.channels.cache.filter(channel => channel.type === 4 && channel.deletable);
        for (const category of categories.values()) {
            try {
                await category.delete();
            } catch (error) {
                console.error(`Kategori silinemedi: ${category.name}`, error);
            }
        }

        const roles = guild.roles.cache.filter(role => 
            role.id !== guild.id && 
            !role.managed && 
            role.editable
        );
        for (const role of roles.values()) {
            try {
                await role.delete();
            } catch (error) {
                console.error(`Rol silinemedi: ${role.name}`, error);
            }
        }

        const emojis = guild.emojis.cache.filter(emoji => emoji.deletable);
        for (const emoji of emojis.values()) {
            try {
                await emoji.delete();
            } catch (error) {
                console.error(`Emoji silinemedi: ${emoji.name}`, error);
            }
        }
    },

    async restoreGuildSettings(guild, guildData) {
        try {
            const updateData = {};
            
            if (guildData.name && guildData.name !== guild.name) {
                updateData.name = guildData.name;
            }
            
            if (guildData.description !== undefined) {
                updateData.description = guildData.description;
            }

            if (guildData.verificationLevel !== undefined) {
                updateData.verificationLevel = guildData.verificationLevel;
            }

            if (guildData.defaultMessageNotifications !== undefined) {
                updateData.defaultMessageNotifications = guildData.defaultMessageNotifications;
            }

            if (guildData.explicitContentFilter !== undefined) {
                updateData.explicitContentFilter = guildData.explicitContentFilter;
            }

            if (Object.keys(updateData).length > 0) {
                await guild.edit(updateData);
            }
        } catch (error) {
            console.error('Sunucu ayarları geri yüklenemedi:', error);
        }
    },

    async restoreRoles(guild, rolesData) {
        const roleMap = {};
        const sortedRoles = rolesData.sort((a, b) => a.position - b.position);

        for (const roleData of sortedRoles) {
            try {
                const newRole = await guild.roles.create({
                    name: roleData.name,
                    color: roleData.color,
                    hoist: roleData.hoist,
                    permissions: roleData.permissions,
                    mentionable: roleData.mentionable,
                    icon: roleData.icon,
                    unicodeEmoji: roleData.unicodeEmoji
                });

                roleMap[roleData.id] = newRole.id;
            } catch (error) {
                console.error(`Rol oluşturulamadı: ${roleData.name}`, error);
            }
        }

        return roleMap;
    },

    async restoreChannels(guild, channelsData) {
        const channelMap = {};
        
        const categories = channelsData.filter(channel => channel.type === 4);
        for (const categoryData of categories) {
            try {
                const newCategory = await guild.channels.create({
                    name: categoryData.name,
                    type: categoryData.type,
                    permissionOverwrites: this.convertPermissionOverwrites(categoryData.permissionOverwrites, {}, {})
                });

                channelMap[categoryData.id] = newCategory.id;
            } catch (error) {
                console.error(`Kategori oluşturulamadı: ${categoryData.name}`, error);
            }
        }

        const otherChannels = channelsData.filter(channel => channel.type !== 4);
        for (const channelData of otherChannels) {
            try {
                const channelOptions = {
                    name: channelData.name,
                    type: channelData.type,
                    parent: channelMap[channelData.parentId] || null,
                    permissionOverwrites: this.convertPermissionOverwrites(channelData.permissionOverwrites, {}, {})
                };

                if (channelData.topic) channelOptions.topic = channelData.topic;
                if (channelData.nsfw !== undefined) channelOptions.nsfw = channelData.nsfw;
                if (channelData.rateLimitPerUser) channelOptions.rateLimitPerUser = channelData.rateLimitPerUser;
                if (channelData.bitrate) channelOptions.bitrate = channelData.bitrate;
                if (channelData.userLimit) channelOptions.userLimit = channelData.userLimit;

                const newChannel = await guild.channels.create(channelOptions);
                channelMap[channelData.id] = newChannel.id;
            } catch (error) {
                console.error(`Kanal oluşturulamadı: ${channelData.name}`, error);
            }
        }

        return channelMap;
    },

    async restoreUserRoles(guild, userRolesData, roleMap) {
        let restoredCount = 0;

        for (const [userId, roleIds] of Object.entries(userRolesData)) {
            try {
                const member = await guild.members.fetch(userId).catch(() => null);
                if (!member) continue;

                const newRoleIds = roleIds
                    .map(roleId => roleMap[roleId])
                    .filter(roleId => roleId && guild.roles.cache.has(roleId));

                if (newRoleIds.length > 0) {
                    await member.roles.set(newRoleIds);
                    restoredCount++;
                }
            } catch (error) {
                console.error(`Kullanıcı rolleri atanamadı: ${userId}`, error);
            }
        }

        return restoredCount;
    },

    async restoreEmojis(guild, emojisData) {
        let restoredCount = 0;

        for (const emojiData of emojisData) {
            try {
                await guild.emojis.create({
                    attachment: emojiData.url,
                    name: emojiData.name
                });
                restoredCount++;
            } catch (error) {
                console.error(`Emoji oluşturulamadı: ${emojiData.name}`, error);
            }
        }

        return restoredCount;
    },

    convertPermissionOverwrites(overwrites, roleMap, channelMap) {
        return overwrites.map(overwrite => ({
            id: roleMap[overwrite.id] || channelMap[overwrite.id] || overwrite.id,
            type: overwrite.type,
            allow: overwrite.allow,
            deny: overwrite.deny
        }));
    }


};