import tkinter as tk
from tkinter import filedialog, messagebox
import os
import re

def select_input_file():
    """Kullanıcıdan giriş JS dosyasını seçtir"""
    file_path = filedialog.askopenfilename(
        title="Temizlenecek JS Dosyasını Seçin",
        filetypes=[("JavaScript Dosyaları", "*.js"), ("Tüm Dosyalar", "*.*")]
    )
    return file_path

def select_output_file(input_path):
    """Çıkış dosyasını kaydetme konumu seçtir (orijinal dosya adının sonuna _clean ekler)"""
    dir_name = os.path.dirname(input_path)
    base_name = os.path.basename(input_path)
    name, ext = os.path.splitext(base_name)
    default_name = f"{name}{ext}"
    default_path = os.path.join(dir_name, default_name)

    file_path = filedialog.asksaveasfilename(
        title="Temizlenmiş Dosyayı Kaydet",
        initialfile=default_name,
        initialdir=dir_name,
        defaultextension=".js",
        filetypes=[("JavaScript Dosyaları", "*.js"), ("Tüm Dosyalar", "*.*")]
    )
    return file_path

def clean_js_file(input_path, output_path):
    """// LOG: ile başlayan satırları sil ve yeni dosyaya yaz"""
    log_pattern = re.compile(r'^\s*// .*$')

    try:
        with open(input_path, 'r', encoding='utf-8') as infile:
            lines = infile.readlines()

        cleaned_lines = []
        for line in lines:
            # Satır // LOG: ile başlıyorsa atla
            if not log_pattern.match(line):
                cleaned_lines.append(line)

        with open(output_path, 'w', encoding='utf-8') as outfile:
            outfile.writelines(cleaned_lines)

        return True, len(lines), len(cleaned_lines)
    except Exception as e:
        return False, str(e), None

def main():
    root = tk.Tk()
    root.withdraw()  # Ana pencereyi gizle

    # 1. Giriş dosyasını seç
    input_path = select_input_file()
    if not input_path:
        messagebox.showinfo("İptal", "Dosya seçilmedi. İşlem iptal edildi.")
        return

    # 2. Çıkış dosyasını seç
    output_path = select_output_file(input_path)
    if not output_path:
        messagebox.showinfo("İptal", "Kaydetme konumu seçilmedi. İşlem iptal edildi.")
        return

    # 3. Temizleme işlemi
    success, total_or_error, cleaned_count = clean_js_file(input_path, output_path)

    if success:
        removed = total_or_error - cleaned_count
        messagebox.showinfo(
            "Başarılı",
            f"Temizleme tamamlandı!\n\n"
            f"📄 Orijinal: {os.path.basename(input_path)}\n"
            f"✨ Temizlenmiş: {os.path.basename(output_path)}\n"
            f"🗑️ Silinen satır: {removed} adet (// LOG: ...)"
        )
    else:
        messagebox.showerror("Hata", f"Dosya işlenirken hata oluştu:\n{total_or_error}")

if __name__ == "__main__":
    main()