const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('whitelist-ekle')
        .setDescription('Discord ID\'yi whitelist\'e ekler')
        .addStringOption(option =>
            option.setName('discordid')
                .setDescription('Whitelist\'e eklenecek Discord ID')
                .setRequired(true)),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'whitelistver')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const discordId = interaction.options.getString('discordid');
            const addedBy = interaction.user.tag;

            // API isteği için body hazırla
            const requestBody = {
                discordId: discordId,
                addedBy: addedBy
            };

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/addwhitelist`;
            const response = await axios.post(apiUrl, requestBody, {
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.data.success) {
                const result = response.data.result;
                const successEmbed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('✅ Whitelist\'e Eklendi')
                    .setDescription(`Discord ID **${discordId}** başarıyla whitelist'e eklendi!`)
                    .addFields({
                        name: '🆔 Discord ID',
                        value: discordId,
                        inline: true
                    })
                    .setFooter({ text: `RoveBot's • Whitelist Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [successEmbed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'whitelistver', {
                    color: 0x00ff00,
                    title: '✅ Whitelist Ekleme Logu',
                    description: `Bir Discord ID whitelist'e eklendi`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '🕐 Zaman',
                            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Whitelist\'e eklenirken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'whitelistver', {
                    color: 0xff0000,
                    title: '❌ Whitelist Ekleme Hatası',
                    description: `Whitelist ekleme işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Whitelist-ekle komutunda hata:', error);
            
            let errorMessage = 'Whitelist\'e eklenirken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'whitelistver', {
                color: 0xff0000,
                title: '❌ Whitelist Ekleme Sistem Hatası',
                description: `Whitelist ekleme işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

