const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('unban')
        .setDescription('Oyuncunun banını kaldırır')
        .addStringOption(option =>
            option.setName('discordid')
                .setDescription('Banı kaldırılacak oyuncunun Discord ID\'si')
                .setRequired(true))
        .addStringOption(option =>
            option.setName('sebep')
                .setDescription('Unban sebebi')
                .setRequired(true)),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'unban')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const discordId = interaction.options.getString('discordid');
            const sebep = interaction.options.getString('sebep');
            const unbannedBy = interaction.user.tag;

            // API isteği için body hazırla
            const requestBody = {
                discordId: discordId,
                unbanReason: sebep,
                unbannedBy: unbannedBy
            };

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/unban`;
            const response = await axios.post(apiUrl, requestBody, {
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.data.success) {
                const result = response.data.result;
                const successEmbed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('✅ Ban Kaldırıldı')
                    .setDescription(`Discord ID **${discordId}**'nin banı başarıyla kaldırıldı!`)
                    .addFields(
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '📝 Sebep',
                            value: sebep,
                            inline: false
                        }
                    )
                    .setFooter({ text: `RoveBot's • Unban Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [successEmbed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'unban', {
                    color: 0x00ff00,
                    title: '✅ Unban Logu',
                    description: `Bir oyuncunun banı kaldırıldı`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '📝 Sebep',
                            value: sebep,
                            inline: false
                        },
                        {
                            name: '🕐 Zaman',
                            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Ban kaldırılırken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'unban', {
                    color: 0xff0000,
                    title: '❌ Unban Hatası',
                    description: `Unban işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Unban komutunda hata:', error);
            
            let errorMessage = 'Ban kaldırılırken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'unban', {
                color: 0xff0000,
                title: '❌ Unban Sistem Hatası',
                description: `Unban işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

