const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('oyuncu-bilgi')
        .setDescription('Oyuncunun bilgilerini görüntüler')
        .addIntegerOption(option =>
            option.setName('oyuncuid')
                .setDescription('Oyuncunun sunucu ID\'si')
                .setRequired(true)
                .setMinValue(1)),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'oyuncubilgi')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const oyuncuId = interaction.options.getInteger('oyuncuid');

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/info`;
            const response = await axios.get(apiUrl, {
                params: {
                    playerId: oyuncuId
                },
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`
                }
            });

            if (response.data.success) {
                const playerInfo = response.data.playerInfo;
                
                // Table formatında bilgileri hazırla
                let infoTable = '```\n';
                infoTable += 'Bilgi'.padEnd(20) + '| Değer\n';
                infoTable += '-'.repeat(20) + '|' + '-'.repeat(40) + '\n';
                
                const fullName = playerInfo.fullName || `${playerInfo.firstName || ''} ${playerInfo.lastName || ''}`.trim() || 'Bilinmiyor';
                infoTable += 'İsim'.padEnd(20) + '| ' + fullName.substring(0, 39) + '\n';
                infoTable += 'Oyuncu ID'.padEnd(20) + '| ' + oyuncuId.toString() + '\n';
                
                if (playerInfo.job) {
                    const jobInfo = `${playerInfo.job.label || playerInfo.job.name || 'Bilinmiyor'} (Seviye: ${playerInfo.job.grade || 0})`;
                    infoTable += 'Meslek'.padEnd(20) + '| ' + jobInfo.substring(0, 39) + '\n';
                }
                
                if (playerInfo.money) {
                    infoTable += 'Nakit'.padEnd(20) + '| $' + (playerInfo.money.cash?.toLocaleString() || '0') + '\n';
                    infoTable += 'Banka'.padEnd(20) + '| $' + (playerInfo.money.bank?.toLocaleString() || '0') + '\n';
                }
                
                infoTable += '```';

                // Identifiers table
                let identifiersTable = '';
                if (playerInfo.identifiers && playerInfo.identifiers.length > 0) {
                    identifiersTable = '```\n';
                    identifiersTable += 'Tip'.padEnd(15) + '| Değer\n';
                    identifiersTable += '-'.repeat(15) + '|' + '-'.repeat(45) + '\n';
                    
                    playerInfo.identifiers.forEach(identifier => {
                        const parts = identifier.split(':');
                        const type = parts[0] || 'Bilinmiyor';
                        const value = parts.slice(1).join(':') || 'Yok';
                        identifiersTable += type.padEnd(15) + '| ' + value.substring(0, 44) + '\n';
                    });
                    
                    identifiersTable += '```';
                } else {
                    identifiersTable = '```\nIdentifiers bulunamadı\n```';
                }

                const embed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('👤 Oyuncu Bilgisi')
                    .setDescription(`Oyuncu ID **${oyuncuId}** bilgileri`)
                    .addFields({
                        name: '📋 Oyuncu Bilgileri',
                        value: infoTable.length > 1024 
                            ? infoTable.substring(0, 1020) + '...```' 
                            : infoTable,
                        inline: false
                    })
                    .addFields({
                        name: '🆔 Identifiers',
                        value: identifiersTable.length > 1024 
                            ? identifiersTable.substring(0, 1020) + '...```' 
                            : identifiersTable,
                        inline: false
                    })
                    .setFooter({ text: `RoveBot's • Oyuncu Bilgi Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [embed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'oyuncubilgi', {
                    color: 0x00ff00,
                    title: '👤 Oyuncu Bilgi Logu',
                    description: `Bir oyuncunun bilgisi görüntülendi`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '👤 Oyuncu İsmi',
                            value: playerInfo.fullName || `${playerInfo.firstName || ''} ${playerInfo.lastName || ''}`.trim() || 'Bilinmiyor',
                            inline: true
                        },
                        {
                            name: '💼 Meslek',
                            value: playerInfo.job ? `${playerInfo.job.label || playerInfo.job.name || 'Bilinmiyor'} (Seviye: ${playerInfo.job.grade || 0})` : 'Yok',
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Oyuncu bilgisi görüntülenirken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'oyuncubilgi', {
                    color: 0xff0000,
                    title: '❌ Oyuncu Bilgi Hatası',
                    description: `Oyuncu bilgisi görüntüleme işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Oyuncu-bilgi komutunda hata:', error);
            
            let errorMessage = 'Oyuncu bilgisi görüntülenirken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'oyuncubilgi', {
                color: 0xff0000,
                title: '❌ Oyuncu Bilgi Sistem Hatası',
                description: `Oyuncu bilgisi görüntüleme işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

