const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('meslek-ver')
        .setDescription('Oyuncuya meslek verir')
        .addIntegerOption(option =>
            option.setName('oyuncuid')
                .setDescription('Oyuncunun sunucu ID\'si')
                .setRequired(true)
                .setMinValue(1))
        .addStringOption(option =>
            option.setName('meslek')
                .setDescription('Meslek adı (örn: police, ambulance)')
                .setRequired(true))
        .addIntegerOption(option =>
            option.setName('seviye')
                .setDescription('Meslek seviyesi (grade)')
                .setRequired(true)
                .setMinValue(0)),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'meslekver')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const oyuncuId = interaction.options.getInteger('oyuncuid');
            const meslek = interaction.options.getString('meslek');
            const seviye = interaction.options.getInteger('seviye');

            // API isteği için body hazırla
            const requestBody = {
                playerId: oyuncuId,
                jobName: meslek,
                jobGrade: seviye
            };

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/setjob`;
            const response = await axios.post(apiUrl, requestBody, {
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.data.success) {
                const result = response.data.result;
                const successEmbed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('✅ Meslek Verildi')
                    .setDescription(`Oyuncu ID **${oyuncuId}**'ye meslek başarıyla verildi!`)
                    .addFields(
                        {
                            name: '💼 Meslek',
                            value: result.job || meslek,
                            inline: true
                        },
                        {
                            name: '📊 Seviye',
                            value: result.grade?.toString() || seviye.toString(),
                            inline: true
                        },
                        {
                            name: '👤 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        }
                    )
                    .setFooter({ text: `RoveBot's • Meslek Verme Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [successEmbed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'meslekver', {
                    color: 0x00ff00,
                    title: '💼 Meslek Verme Logu',
                    description: `Bir oyuncuya meslek verildi`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '💼 Meslek',
                            value: result.job || meslek,
                            inline: true
                        },
                        {
                            name: '📊 Seviye',
                            value: result.grade?.toString() || seviye.toString(),
                            inline: true
                        },
                        {
                            name: '🕐 Zaman',
                            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Meslek verilirken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'meslekver', {
                    color: 0xff0000,
                    title: '❌ Meslek Verme Hatası',
                    description: `Meslek verme işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '💼 Meslek',
                            value: meslek,
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Meslek-ver komutunda hata:', error);
            
            let errorMessage = 'Meslek verilirken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'meslekver', {
                color: 0xff0000,
                title: '❌ Meslek Verme Sistem Hatası',
                description: `Meslek verme işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

