const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('ban-bilgi')
        .setDescription('Oyuncunun ban bilgisini görüntüler')
        .addStringOption(option =>
            option.setName('discordid')
                .setDescription('Oyuncunun Discord ID\'si')
                .setRequired(true)),

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'banbilgi')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const discordId = interaction.options.getString('discordid');

            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/baninfo`;
            const response = await axios.get(apiUrl, {
                params: {
                    discordId: discordId
                },
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`
                }
            });

            if (response.data.success) {
                const banInfo = response.data.banInfo;
                
                const embed = new EmbedBuilder()
                    .setColor(banInfo.isUnbanned ? 0x00ff00 : 0xff0000)
                    .setTitle('🚫 Ban Bilgisi')
                    .setDescription(`Discord ID **${discordId}** ban bilgisi`)
                    .addFields(
                        {
                            name: '🆔 Discord ID',
                            value: banInfo.discordId,
                            inline: true
                        },
                        {
                            name: '📝 Ban Sebebi',
                            value: banInfo.banReason || 'Belirtilmemiş',
                            inline: true
                        },
                        {
                            name: '⏱️ Ban Süresi',
                            value: banInfo.banDuration || 'Bilinmiyor',
                            inline: true
                        },
                        {
                            name: '👤 Banlayan',
                            value: banInfo.bannedBy || 'Bilinmiyor',
                            inline: true
                        },
                        {
                            name: '📅 Ban Tarihi',
                            value: banInfo.bannedAt 
                                ? `<t:${Math.floor(new Date(banInfo.bannedAt).getTime() / 1000)}:F>`
                                : 'Bilinmiyor',
                            inline: true
                        },
                        {
                            name: '✅ Durum',
                            value: banInfo.isUnbanned ? '❌ Ban Kaldırıldı' : '🚫 Banlı',
                            inline: true
                        }
                    )
                    .setFooter({ text: `RoveBot's • Ban Bilgi Sistemi` })
                    .setTimestamp();

                // Eğer ban kaldırıldıysa unban bilgilerini ekle
                if (banInfo.isUnbanned) {
                    embed.addFields({
                        name: '🔓 Unban Bilgileri',
                        value: `**Unbanlayan:** ${banInfo.unbannedBy || 'Bilinmiyor'}\n**Sebep:** ${banInfo.unbanReason || 'Belirtilmemiş'}\n**Tarih:** ${banInfo.unbannedAt ? `<t:${Math.floor(new Date(banInfo.unbannedAt).getTime() / 1000)}:F>` : 'Bilinmiyor'}`,
                        inline: false
                    });
                } else {
                    // Ban süresi bilgilerini ekle
                    if (banInfo.expiresAt) {
                        embed.addFields({
                            name: '⏳ Bitiş Tarihi',
                            value: `<t:${banInfo.expiresAt}:F>`,
                            inline: true
                        });
                    }
                    
                    if (banInfo.remainingTime && !banInfo.isExpired) {
                        embed.addFields({
                            name: '⏰ Kalan Süre',
                            value: banInfo.remainingTime,
                            inline: true
                        });
                    }
                    
                    if (banInfo.isExpired) {
                        embed.addFields({
                            name: '⚠️ Durum',
                            value: 'Ban süresi dolmuş',
                            inline: true
                        });
                    }
                }

                await interaction.editReply({ embeds: [embed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'banbilgi', {
                    color: banInfo.isUnbanned ? 0x00ff00 : 0xff0000,
                    title: '📋 Ban Bilgi Logu',
                    description: `Bir oyuncunun ban bilgisi görüntülendi`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '📝 Ban Sebebi',
                            value: banInfo.banReason || 'Belirtilmemiş',
                            inline: true
                        },
                        {
                            name: '⏱️ Ban Süresi',
                            value: banInfo.banDuration || 'Bilinmiyor',
                            inline: true
                        },
                        {
                            name: '✅ Durum',
                            value: banInfo.isUnbanned ? 'Ban Kaldırıldı' : 'Banlı',
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Ban bilgisi görüntülenirken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'banbilgi', {
                    color: 0xff0000,
                    title: '❌ Ban Bilgi Hatası',
                    description: `Ban bilgisi görüntüleme işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🆔 Discord ID',
                            value: discordId,
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Ban-bilgi komutunda hata:', error);
            
            let errorMessage = 'Ban bilgisi görüntülenirken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed] });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'banbilgi', {
                color: 0xff0000,
                title: '❌ Ban Bilgi Sistem Hatası',
                description: `Ban bilgisi görüntüleme işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

