const { SlashCommandBuilder, EmbedBuilder } = require('discord.js');
const { checkPermission } = require('./yetkiayarla');
const { sendLog } = require('../utils/logSystem');
const config = require('../../config.json');
const axios = require('axios');

module.exports = {
    data: new SlashCommandBuilder()
        .setName('arac-ver')
        .setDescription('Oyuncuya araç verir')
        .addIntegerOption(option =>
            option.setName('oyuncuid')
                .setDescription('Oyuncunun sunucu ID\'si')
                .setRequired(true)
                .setMinValue(1))
        .addStringOption(option =>
            option.setName('aracadi')
                .setDescription('Araç model adı (örn: adder, sultan)')
                .setRequired(true))
        .addStringOption(option =>
            option.setName('plaka')
                .setDescription('Özel plaka (boş bırakılırsa otomatik oluşturulur)')
                .setRequired(false)),
        

    async execute(interaction) {
        try {
            // Yetki kontrolü
            if (!checkPermission(interaction, 'aracver')) {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Yetki Hatası')
                    .setDescription('Bu komutu kullanmak için gerekli yetkiye sahip değilsiniz!')
                    .setTimestamp();

                return await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            await interaction.deferReply();

            const oyuncuId = interaction.options.getInteger('oyuncuid');
            const aracAdi = interaction.options.getString('aracadi');
            const plaka = interaction.options.getString('plaka');
            const fullmod = false;

            // API isteği için body hazırla
            const requestBody = {
                playerId: oyuncuId,
                carName: aracAdi
            };

            if (plaka) {
                requestBody.plate = plaka;
            }

            if (fullmod) {
                requestBody.fullmod = fullmod;
            }


            // API isteği gönder
            const apiUrl = `http://${config.BotKurulumu.vdsip}/api/server/player/givevehicle`;
            const response = await axios.post(apiUrl, requestBody, {
                headers: {
                    'Authorization': `Bearer ${config.BotKurulumu.apikey}`,
                    'Content-Type': 'application/json'
                }
            });

            if (response.data.success) {
                const result = response.data.result;
                const successEmbed = new EmbedBuilder()
                    .setColor(0x00ff00)
                    .setTitle('✅ Araç Verildi')
                    .setDescription(`Oyuncu ID **${oyuncuId}**'ye araç başarıyla verildi!`)
                    .addFields(
                        {
                            name: '🚗 Araç Modeli',
                            value: result.vehicle || aracAdi,
                            inline: true
                        },
                        {
                            name: '🔖 Plaka',
                            value: result.plate || 'Otomatik oluşturuldu',
                            inline: true
                        },
                        {
                            name: '👤 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        }
                    )
                    .setFooter({ text: `RoveBot's • Araç Verme Sistemi` })
                    .setTimestamp();

                await interaction.editReply({ embeds: [successEmbed] });

                // Log gönder
                await sendLog(interaction.client, interaction.guild.id, 'aracver', {
                    color: 0x00ff00,
                    title: '🚗 Araç Verme Logu',
                    description: `Bir oyuncuya araç verildi`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '🚗 Araç Modeli',
                            value: result.vehicle || aracAdi,
                            inline: true
                        },
                        {
                            name: '🔖 Plaka',
                            value: result.plate || 'Otomatik oluşturuldu',
                            inline: true
                        },
                        
                        {
                            name: '🕐 Zaman',
                            value: `<t:${Math.floor(Date.now() / 1000)}:F>`,
                            inline: true
                        }
                    ]
                });

            } else {
                const errorEmbed = new EmbedBuilder()
                    .setColor(0xff0000)
                    .setTitle('❌ Hata')
                    .setDescription(response.data.error || 'Araç verilirken bir hata oluştu!')
                    .setTimestamp();

                await interaction.editReply({ embeds: [errorEmbed] });

                // Hata logu gönder
                await sendLog(interaction.client, interaction.guild.id, 'aracver', {
                    color: 0xff0000,
                    title: '❌ Araç Verme Hatası',
                    description: `Araç verme işlemi başarısız oldu`,
                    user: interaction.user,
                    fields: [
                        {
                            name: '👤 Yetkili',
                            value: `${interaction.user} (${interaction.user.tag})`,
                            inline: true
                        },
                        {
                            name: '🎮 Oyuncu ID',
                            value: oyuncuId.toString(),
                            inline: true
                        },
                        {
                            name: '🚗 Araç Modeli',
                            value: aracAdi,
                            inline: true
                        },
                        {
                            name: '❌ Hata',
                            value: response.data.error || 'Bilinmeyen hata',
                            inline: false
                        }
                    ]
                });
            }

        } catch (error) {
            console.error('Arac-ver komutunda hata:', error);
            
            let errorMessage = 'Araç verilirken bir hata oluştu!';
            
            if (error.response) {
                // API'den gelen hata
                errorMessage = error.response.data?.error || errorMessage;
            } else if (error.message) {
                errorMessage = error.message;
            }

            const errorEmbed = new EmbedBuilder()
                .setColor(0xff0000)
                .setTitle('❌ Sistem Hatası')
                .setDescription(errorMessage)
                .setTimestamp();

            if (interaction.deferred) {
                await interaction.editReply({ embeds: [errorEmbed] });
            } else {
                await interaction.reply({ embeds: [errorEmbed], ephemeral: true });
            }

            // Hata logu gönder
            await sendLog(interaction.client, interaction.guild.id, 'aracver', {
                color: 0xff0000,
                title: '❌ Araç Verme Sistem Hatası',
                description: `Araç verme işlemi sırasında sistem hatası oluştu`,
                user: interaction.user,
                fields: [
                    {
                        name: '👤 Yetkili',
                        value: `${interaction.user} (${interaction.user.tag})`,
                        inline: true
                    },
                    {
                        name: '❌ Hata Detayı',
                        value: errorMessage,
                        inline: false
                    }
                ]
            });
        }
    }
};

