const { Client, GatewayIntentBits, Collection, REST, Routes } = require('discord.js');
const fs = require('fs');
const path = require('path');
const config = require('./config.json');
const KeepAlive = require('./src/keepalive');

const client = new Client({
    intents: [
        GatewayIntentBits.Guilds,
        GatewayIntentBits.GuildMessages,
        GatewayIntentBits.MessageContent,
        GatewayIntentBits.GuildMembers,
        GatewayIntentBits.GuildVoiceStates,
        GatewayIntentBits.GuildBans,
        GatewayIntentBits.GuildInvites
    ]
});

client.commands = new Collection();

const commandsPath = path.join(__dirname, 'src', 'commands');
if (fs.existsSync(commandsPath)) {
    const commandFiles = fs.readdirSync(commandsPath).filter(file => file.endsWith('.js'));
    
    for (const file of commandFiles) {
        const filePath = path.join(commandsPath, file);
        const command = require(filePath);
        
        if ('data' in command && 'execute' in command) {
            client.commands.set(command.data.name, command);
            console.log(`[KOMUT] ${command.data.name} komutu yüklendi.`);
        } else {
            console.log(`[UYARI] ${filePath} komutunda gerekli "data" veya "execute" özelliği eksik.`);
        }
    }
}

const eventsPath = path.join(__dirname, 'src', 'events');
const eventFiles = fs.readdirSync(eventsPath).filter(file => file.endsWith('.js'));

for (const file of eventFiles) {
    const filePath = path.join(eventsPath, file);
    const event = require(filePath);
    
    if (event.once) {
        client.once(event.name, (...args) => event.execute(...args));
    } else {
        client.on(event.name, (...args) => event.execute(...args));
    }
    console.log(`[EVENT] ${event.name} eventi yüklendi.`);
}

async function deployCommands() {
    const commands = [];
    
    if (fs.existsSync(commandsPath)) {
        const commandFiles = fs.readdirSync(commandsPath).filter(file => file.endsWith('.js'));
        
        for (const file of commandFiles) {
            const filePath = path.join(commandsPath, file);
            const command = require(filePath);
            if ('data' in command) {
                commands.push(command.data.toJSON());
            }
        }
    }
    
    const rest = new REST().setToken(config.BotKurulumu.token);
    
    try {
        console.log(`${commands.length} adet slash komutu kaydediliyor...`);
        
        const data = await rest.put(
            Routes.applicationGuildCommands(config.BotKurulumu.application, config.BotKurulumu.guild),
            { body: commands },
        );
        
        console.log(`${data.length} adet slash komutu başarıyla kaydedildi.`);
    } catch (error) {
        console.error('Slash komutları kaydedilirken hata oluştu:', error);
    }
}

async function startBot() {
    try {
        const keepAlive = new KeepAlive(9676);
        await keepAlive.start();
        keepAlive.setupGracefulShutdown();
        
        await client.login(config.BotKurulumu.token);
        console.log('🤖 Discord botu başarıyla başlatıldı!');

        client.on('error', (error) => {
            console.error('❌ Discord client hatası:', error);
        });
        
        client.on('warn', (warning) => {
            console.warn('⚠️ Discord client uyarısı:', warning);
        });
        
        client.on('disconnect', (event) => {
            console.warn('🔌 Discord bağlantısı kesildi:', event);
        });
        
        client.on('reconnecting', () => {
            console.log('🔄 Discord\'a yeniden bağlanılıyor...');
        });
        
        client.on('resume', (replayed) => {
            console.log(`✅ Discord bağlantısı yeniden kuruldu. ${replayed} event tekrarlandı.`);
        });
        
        client.once('ready', async () => {
            setInterval(async () => {
                const memUsage = process.memoryUsage();
                const heapUsedMB = Math.round(memUsage.heapUsed / 1024 / 1024);
                const memoryThreshold = 512;
                
                if (heapUsedMB > memoryThreshold) {
                    console.error(`🚨 KRİTİK: Bellek kullanımı çok yüksek! ${heapUsedMB}MB / ${memoryThreshold}MB`);
                    
                    if (client.guilds && client.guilds.cache) {
                        client.guilds.cache.forEach(guild => {
                            if (guild.channels && guild.channels.cache) {
                                guild.channels.cache.forEach(channel => {
                                    if (channel.messages && channel.messages.cache.size > 10) {
                                        const messages = Array.from(channel.messages.cache.values());
                                        const toKeep = messages.slice(-10);
                                        channel.messages.cache.clear();
                                        toKeep.forEach(msg => channel.messages.cache.set(msg.id, msg));
                                    }
                                });
                            }
                        });
                    }
                    
                    if (global.gc) {
                        global.gc();
                        console.log('🧹 Acil garbage collection tetiklendi');
                    }
                    
                    const newMemUsage = Math.round(process.memoryUsage().heapUsed / 1024 / 1024);
                    if (newMemUsage > memoryThreshold * 0.9) {
                        console.error(`⚠️ UYARI: Bellek kullanımı hala yüksek: ${newMemUsage}MB`);
                        console.error('💡 Bot yeniden başlatılması önerilir!');
                    }
                }
                
                if (client.ws.ping > 1000) {
                    console.warn(`⚠️ Yüksek ping tespit edildi: ${client.ws.ping}ms`);
                }
                
                const uptimeHours = process.uptime() / 3600;
                if (uptimeHours > 24) {
                    console.log(`📊 Bot ${Math.floor(uptimeHours)} saattir çalışıyor. Bellek: ${heapUsedMB}MB`);
                }
            }, 5 * 60 * 1000);
        });
        
        
        setInterval(() => {
            try {
                console.log('🧹 Discord cache temizliği başlatılıyor...');
                
                client.guilds.cache.forEach(guild => {
                    guild.channels.cache.forEach(channel => {
                        if (channel.messages) {
                            const messageCount = channel.messages.cache.size;
                            if (messageCount > 100) {
                                const messages = Array.from(channel.messages.cache.values())
                                    .sort((a, b) => b.createdTimestamp - a.createdTimestamp)
                                    .slice(50);
                                
                                messages.forEach(msg => channel.messages.cache.delete(msg.id));
                                console.log(`📝 ${channel.name} kanalından ${messages.length} mesaj cache'den temizlendi`);
                            }
                        }
                    });
                });
                
                client.guilds.cache.forEach(guild => {
                    const presenceCount = guild.presences.cache.size;
                    if (presenceCount > 500) {
                        guild.presences.cache.clear();
                        console.log(`👥 ${guild.name} sunucusundan ${presenceCount} presence cache'den temizlendi`);
                    }
                });
                
                client.guilds.cache.forEach(guild => {
                    const voiceStates = guild.voiceStates.cache;
                    const emptyVoiceStates = voiceStates.filter(vs => !vs.channelId);
                    emptyVoiceStates.forEach(vs => voiceStates.delete(vs.id));
                    
                    if (emptyVoiceStates.size > 0) {
                        console.log(`🔊 ${guild.name} sunucusundan ${emptyVoiceStates.size} boş voice state temizlendi`);
                    }
                });
                
                if (global.gc) {
                    global.gc();
                    console.log('🗑️ Garbage collection tetiklendi');
                }
                
                console.log('✅ Discord cache temizliği tamamlandı');
            } catch (error) {
                console.error('❌ Cache temizliği sırasında hata:', error);
            }
        }, 20 * 60 * 1000);
        
        await deployCommands();
        
    } catch (error) {
        console.error('❌ Bot başlatılırken hata oluştu:', error);
        process.exit(1);
    }
}

startBot();